#!/usr/bin/env python3

import sys
import numpy as np # to install this lib in deb-type linux, run as root: apt-get install python3-numpy
import xyz_utils
from collections import namedtuple
hund80overpi=180/np.pi     # 57.29577951308232

def rotation(vect):
    """return a 3D rotation around vect by angle(in radians)=norm of vect
    using Rodrigues' rotation formula
    see e.g. https://en.wikipedia.org/wiki/Rodrigues%27_rotation_formula"""
    angle=np.linalg.norm(vect)
    cangle=np.cos(angle)
    r=cangle*np.identity(3)
    if angle<1.e-16 :	# practically no rotation
        return r
    v=vect/angle	# normalized to unity
    vx=np.array([[0,-v[2],v[1]],[v[2],0,-v[0]],[-v[1],v[0],0]])
    vp=np.tensordot(v,v,axes=0)
    return r+np.sin(angle)*vx+(1-cangle)*vp

def frame_rotate(fr,rotmatrix=None,ax=None):
    """rotate an entire xyz frame
    fr=xyz frame; 
    rotmatrix=3x3 rotation matrix; 
    ax=rotation axis; its module is the rotation angle in radians"""
    atoms=fr.atoms
    comment=fr.comment
    if np.size(ax)>1:
        comment+=" rotated by "+str(np.linalg.norm(ax)*hund80overpi)+" degrees around "+str(ax)
    if np.size(rotmatrix)==1:
        rotmatrix=rotation(ax)
    coords=fr.coords  # allocate the right memory size
    for i in range(len(coords)): # python gurus may code this better...
        coords[i]=np.dot(rotmatrix,coords[i])
    return namedtuple("xyzdata", ["comment", "atoms", "coords"])(comment, atoms, coords)

def xyz_rotate(args):

    ax=np.array(args.ax)
    if args.vx!= 0.:
        ax[0]=args.vx
    if args.vy!= 0.:
        ax[1]=args.vy
    if args.vz!= 0.:
        ax[2]=args.vz
    no=np.linalg.norm(ax)
    if no>0.:
        ax/=no
#   now ax is normalized to unity

    if args.angledegrees!= 0.:
        angle=args.angledegrees/hund80overpi
    else:
        angle=args.angle

    ax*=angle

    rotmatrix=rotation(ax)  # rotation matrix computed once for all

    if args.debug:
        print("debug: ", rotmatrix)
    for filen in args.filenames:
        if filen=="-":
            f=sys.stdin
        else:
            f = open(filen, 'r')

        while True: # loop on the smapshots of all files
            fr=xyz_utils.xyz_read_one_frame(f)
            if fr.atoms == None:
                break
            else:
                frr=frame_rotate(fr,rotmatrix,ax)
                xyz_utils.xyz_write_one_frame(sys.stdout,frr)
        

# the following function is only exectuted when this code is run as a script, and its purposes is to parse
# the command line and to generate a meaningful parsed args list to the actual function doing the job:
if __name__ == "__main__":
    import argparse
    commandname=sys.argv[0]

    desc="""Rotate xyz files. 
INPUT: various xyz files
OUTPUT: a xyz file """

    epil="""                      v2.2 by N. Manini, 4.05.2020"""

##  Argument Parser definition:
    parser = argparse.ArgumentParser( formatter_class=argparse.ArgumentDefaultsHelpFormatter
                                    , description=desc, epilog=epil)

    parser.add_argument( 'filenames', nargs='*', default=['-'],
                         help='Files to be processed. If not given, stdin is used')

    parser.add_argument( '-r',
                         dest='angle', type=float, default=0.,
                         help='the rotation angle in radians')

    parser.add_argument( '-a',
                         dest='angledegrees', type=float, default=0.,
                         help='the rotation angle in degrees')

    parser.add_argument( '--ax', type=float, nargs=3,
                         dest='ax', default=[0.,0.,1.],
                         help='the rotation axis, expecting 3 components')

    parser.add_argument( '--vx',
                         dest='vx', type=float, default=0.,
                         help='the x component of the rotation axis')

    parser.add_argument( '--vy',
                         dest='vy', type=float, default=0.,
                         help='the y component of the rotation axis')

    parser.add_argument( '--vz',
                         dest='vz', type=float, default=0.,
                         help='the z component of the rotation axis')

    parser.add_argument( '-d', '--debug', action='store_true',
                         dest='debug', 
                         help='activate debug mode -- WARNING! output is affected/spoiled!' )

## End arg parser definition
    args=parser.parse_args(sys.argv[1:])
    d = vars(args)	# adding prog in args, for unknown reasons it's not there...
    d['prog']=parser.prog
#   here the actual function doing the job is called:
    xyz_rotate(args)
