#!/usr/bin/env python3

import sys
import numpy as np
import xyz_utils
from collections import namedtuple


def frame_translate(fr,vt):
    """translate an xyz frame
    fr=xyz frame; vt=translation vector"""
    comment=fr.comment+" translated by "+str(vt)
    atoms=fr.atoms
    coords=fr.coords+vt
    return namedtuple("xyzdata", ["comment", "atoms", "coords"])(comment, atoms, coords)

def xyz_translate(args):

    vt=np.array(args.vtrans)
    if args.xtrans!= 0.:
        vt[0]=args.xtrans
    if args.ytrans!= 0.:
        vt[1]=args.ytrans
    if args.ztrans!= 0.:
        vt[2]=args.ztrans

    for filen in args.filenames:
        if filen=="-":
            f=sys.stdin
        else:
            f = open(filen, 'r')

        while True: # loop on the smapshots of all files
            fr=xyz_utils.xyz_read_one_frame(f)
            if fr.atoms == None:
                break
            else:
                frt=frame_translate(fr,vt)
                xyz_utils.xyz_write_one_frame(sys.stdout,frt)
        

# the following function is only exectuted when this code is run as a script, and its purposes is to parse
# the command line and to generate a meaningful parsed args list to the actual function doing the job:
if __name__ == "__main__":
    import argparse
    commandname=sys.argv[0]

    desc="""Translate xyz files. 
INPUT: various xyz files
OUTPUT: a xyz file """

    epil="""                      v1.1 by N. Manini, 23.05.2020"""

##  Argument Parser definition:
    parser = argparse.ArgumentParser( formatter_class=argparse.ArgumentDefaultsHelpFormatter
                                    , description=desc, epilog=epil)

    parser.add_argument( 'filenames', nargs='*', default=['-'],
                         help='Files to be processed. If not given, stdin is used')

    parser.add_argument( '-x',
                         dest='xtrans', type=float, default=0.,
                         help='the x component of the translation')

    parser.add_argument( '-y',
                         dest='ytrans', type=float, default=0.,
                         help='the y component of the translation')

    parser.add_argument( '-z',
                         dest='ztrans', type=float, default=0.,
                         help='the z component of the translation')

    parser.add_argument( '-v', type=float, nargs=3,
                         dest='vtrans', default=[0.,0.,0.],
                         help='the translation vector, expecting 3 components')

## End arg parser definition
    args=parser.parse_args(sys.argv[1:])
    d = vars(args)	# adding prog in args, for unknown reasons it's not there...
    d['prog']=parser.prog
#   here the actual function doing the job is called:
    xyz_translate(args)
