#!/usr/bin/env python3

import sys
import re
import os
import os.path
import numpy as np # to install this lib in deb-type linux, run as root: apt-get install python3-numpy
import xyz_utils
from collections import namedtuple

def xyz_addrandom(filenames,std=0.1,printout=False,rectangular=False):
    """read files and add random displacement to atomic positions"""
    if len(filenames)<1:
        print("error: need at least 1 file!!!", file=sys.stderr)
        return None
    
    allfr=[]
    for filen in filenames:
        if filen=="-":
            f=sys.stdin
        else:
            f = open(filen, 'r')
        while True:
            frame=xyz_utils.xyz_read_one_frame(f) # read one frame from file
            if frame.atoms == None:
                break
            atoms=frame.atoms
            coords=np.empty_like(frame.coords)
            for it in range(args.nrepeat):
                comment=frame.comment+" randomized "+str(it)
                if rectangular:
                    fact=np.sqrt(12)*std  # with this factor, the standard deviation of the uniform distribution becomes std
                    coords=frame.coords+(np.random.rand(len(atoms),3)-0.5)*fact
                else:
                    coords=frame.coords+np.random.randn(len(atoms),3)*std
                fr=namedtuple("xyzdata", ["comment", "atoms", "coords"])(comment, atoms, coords)
                if printout:
                    xyz_utils.xyz_write_one_frame(sys.stdout,fr)
                else:
                    allfr.append(fr) # memory-dangerous! cumulating all frames in memory...
    return allfr
        

# the following function is only exectuted when this code is run as a script, and its purposes is to parse
# the command line and to generate a meaningful parsed args list to the actual function doing the job:
if __name__ == "__main__":
    import sys
    import argparse
    commandname=sys.argv[0]
# default values:
    filenames = []

    desc="""displace each atomic coordinate by a random number.    
    Can repeat the operation n times
    INPUT: xyz files
    OUTPUT: a xyz file with n times the number of original frames"""

    epil="""			v1.2 by N. Manini, 02.05.2020"""

    ##  Argument Parser definition: this is just an example...
    parser = argparse.ArgumentParser( formatter_class=argparse.ArgumentDefaultsHelpFormatter
                                    , description=desc, epilog=epil)

    parser.add_argument( 'filenames', nargs='*', default=['-'],
                         help='Files to be processed. If not given, stdin is used')

    parser.add_argument( '-s',
                         dest='std', type=float, default=0.1,
                         help='the standard deviation of the random noise')
   
    parser.add_argument( '-r', '--rectangular', action='store_true',
                         dest='rectangular', 
                         help='use a rectangular rather than the normal (Gaussian) distrubution')
   
    parser.add_argument( '-n', type=int, default=1,
                         dest='nrepeat', 
                         help='number of instances per each frame')
   
    ## End arg parser definition
    args=parser.parse_args(sys.argv[1:])
    d = vars(args)	# adding prog in args, for unknown reasons it's not there...
    d['prog']=parser.prog
#   here the actual function doing the job is called:
    a=xyz_addrandom(args.filenames,args.std,True,args.rectangular)
#   final printout:
#   xyz_utils.xyz_write_entire_file(sys.stdout, a)
