#!/usr/bin/env python3

import re
import sys
import scipy.constants 

def multiplyifpossible(a,b):
    """execute the multiplication only in case a is a float, otherwise just return a"""
    if isinstance(a,float):
        return a*b
    else:
        return a

def printall(results,header=False,eVunits=False):
    units="Ry"
    efactor=1.
    if eVunits:
        units="eV"
        efactor=scipy.constants.physical_constants["Hartree energy in eV"][0]/2.
    if header:
        print("#"," ".join(key for key in results.keys())," [in",units,"units]")
    print(" ".join(str(multiplyifpossible(results[key],efactor)) for key in results.keys()))

def extractQEdata(args):
    """select neeeded lines from a QE output file"""
    
    results={}  # create an empty dictionary to store all stuff
    for filen in args.filenames: # loop over the files, open if needed:
        if args.debug:
            print(" debug: args=", args)
        if filen=="-":
            f=sys.stdin
        else:
            f = open(filen, 'r')
        results.update({"filen":filen})

        interestingstuff=False
        for line in f:
            if re.search("convergence has been achieved in",line):
                interestingstuff=False
# print results here if need results at each relaxation step:
                printall(results,False,args.eVunits)

            if interestingstuff: # here the actual processing is made:
                sline=re.sub("^\s+","",line).rstrip() # remove initial spaces and trailing newline
                nums=re.split('\s+',sline) # put the individial fields in nums
                if args.debug:
                    print(" debug ", len(nums), " nums[0]: ",nums[0])
                if re.search("!    total energy",line):
                    toten=float(nums[4])
                    results.update({"toten":toten})
                if re.search("     one-electron contribution",line):
                    oneelen=float(nums[3])
                    results.update({"oneelen":oneelen})
                if re.search("     hartree contribution",line):
                    hartreeen=float(nums[3])
                    results.update({"hartreeen":hartreeen})
                if re.search("     xc contribution",line):
                    xcen=float(nums[3])
                    results.update({"xcen":xcen})
                if re.search("     ewald contributio",line):
                    ewalden=float(nums[3])
                    results.update({"ewalden":ewalden})
                if re.search("     smearing contrib",line):
                    smearingen=float(nums[4])
                    results.update({"smearingen":smearingen})
 # potentially collect more info, e.g. Fermi energy, magnetization...
 
            if re.search("End of self-consistent calculation",line):
                interestingstuff=True

# print results here if need results at the final step only:
        printall(results,True,args.eVunits)
    return results    # return results if useful outside
        

# the following function is only exectuted when this code is run as a script, and its purposes is to parse
# the command line and to generate a meaningful parsed args list for the actual function doing the job:
if __name__ == "__main__":
    import argparse
    commandname=sys.argv[0]

    desc="""Postprocessing of QE output files
INPUT : QE output file(s)
OUTPUT: interesting data, ready for plotting"""

    epil="""v1.2 by N. Manini, 23.04.2020"""

    ##  Argument Parser definition: this is just an example...
    parser = argparse.ArgumentParser( formatter_class=argparse.ArgumentDefaultsHelpFormatter
                                    , description=desc, epilog=epil)

    parser.add_argument( 'filenames', nargs='*', default=['-'],
                         help='Files to be processed. If not given, stdin is used')
    parser.add_argument( '--eV', action='store_true',
                         dest='eVunits', 
                         help='print results in eV units, rather than the default Ry units' )
    parser.add_argument( '-d', '--debug', action='store_true',
                         dest='debug', 
                         help='activate debug mode -- WARNING! output is affected/spoiled!' )

    ## End arg parser definition
    args=parser.parse_args(sys.argv[1:])
    d = vars(args)	# adding prog in args, for unknown reasons it's not there...
    d['prog']=parser.prog
#   here the actual function doing the job is called:
    results=extractQEdata(args)  # NB! we pass args now, not just the filenames
#   final printout:
#    print(results)
