#!/usr/bin/env python3

################################################################################################
#     calculate the average angle of the vector joining molecular cation and anion, relative to the vertical
#     works with a data file containing planted molecules in the sub AND in the sup, taking PBC into account
################################################################################################

import re
import numpy as np

### angle calculation: sign=1 for SUB and -1 for SUP
def angle_calc(cation_position, anion_position, supercell, sign=1):
    vd=cation_position-anion_position # joining vector
    vdm=vd/supercell	# in supercell units
    vdm-=np.rint(vdm)	# now in (-0.5,0.5] range
    vd=vdm*supercell	# joining vector reported to nearest image
    angle=sign*np.arctan(vd[2]/np.linalg.norm(np.take(vd,[0,1])))*180/np.pi+90
    return(angle)

def extract_angle_from_data(args):
    cation=str(args.cation)
    anion=str(args.anion)

    for filen in args.filenames:
        count=0
        if filen=="-":
            f=sys.stdin
        else:
            f = open(filen, 'r')
        zeta = 0
        anions = 0
        supercell=np.zeros(3)
####################### create an empty np array
        arr = np.empty((0,5)) #, dtype=np.float64)
        for line in f:
            line=re.sub("^\s+","",line).rstrip()
            l=re.split('\s+',line)
            if args.debug:
                print ("debug " , len(l), " 0: ",l[0])
            if len(l)>6:
                if l[2]==cation or l[2]==anion:
                    arr = np.append(arr, np.array([[int(l[1]),l[2],float(l[4]),float(l[5]),float(l[6])]]), axis=0)
                    if l[2]==anion:
                        anions += 1
                        zeta += float(l[6])
            else: # collect the box sides:
                if re.search("xlo xhi",line):
                    supercell[0]=float(l[1])-float(l[0])
                if re.search("ylo yhi",line):
                    supercell[1]=float(l[1])-float(l[0])
                if re.search("zlo zhi",line):
                    supercell[2]=float(l[1])-float(l[0])
        if filen!="-":
            f.close()

        if args.debug:
            print ("debug supercell",supercell)


####################### average ANIONS zed coordinate
        zeta_ave = zeta/anions
        if args.debug:
            print ("debug zave,naions",zeta_ave,anions)

################# sort by first column (molecule index):
        arr = arr[np.lexsort((arr[:,0], arr[:,0]))]

        if args.debug:
            print ("debug sorted arr:",arr)

#    resetting counters:
        ave_angleSUB = 0.
        ave_angle2SUB = 0.
        molecSUB = 0
        ave_angleSUP = 0.
        ave_angle2SUP = 0.
        molecSUP = 0
        cati = False
        anio = False
        sign=1
        
        for lista in arr:
            if lista[1] == cation:
                cati = True
                posc=np.array([float(i) for i in lista[2:5]])
            if lista[1] == anion:
                anio = True
                posa=np.array([float(i) for i in lista[2:5]])
                if float(lista[4]) < zeta_ave: # SUB molecules
                    sign=1
                else:                          # SUP molecules
                    sign=-1
            if cati and anio:
                angle=angle_calc(posc,posa,supercell,sign)
                if sign==1:   # SUB molecules
                    ave_angleSUB += angle
                    ave_angle2SUB += angle*angle
                    molecSUB += 1
                else:        # SUP molecules
                    ave_angleSUP += angle
                    ave_angle2SUP += angle*angle
                    molecSUP += 1
                cati = False
                anio = False
        ave_angleSUB=ave_angleSUB/molecSUB
        ave_angle2SUB=ave_angle2SUB/molecSUB
        std_angleSUB=np.sqrt(ave_angle2SUB-ave_angleSUB*ave_angleSUB)
        ave_angleSUP=ave_angleSUP/molecSUP
        ave_angle2SUP=ave_angle2SUP/molecSUP
        std_angleSUP=np.sqrt(ave_angle2SUP-ave_angleSUP*ave_angleSUP)
        
        if args.header:
            print ('#SUB:average      std              nmol  SUP:average       std              nmol  filename')
        print (ave_angleSUB,std_angleSUB,molecSUB,ave_angleSUP,std_angleSUP,molecSUP,filen)



# the following function is only exectuted when the code is run as a script, and its purposes is to parse
# the command line and to generate a meaningful parsed args list for the actual function doing the job:
if __name__ == "__main__":
    import sys
    import argparse
    commandname=sys.argv[0]
# default values:
    filenames = []

    desc="""compute the average angle of the cation and anion relative to the vertical direction.
        Works with a lammps data file containing planted molecules in the sub & in the sup
INPUT:  lammps data file
OUTPUT: angle statistics"""

    epil="""originally written by M.M. Gianetti; v. 3.0 by Nicola Manini, 20/04/2020"""

    ##  Argument Parser definition: this is just an example...
    parser = argparse.ArgumentParser( formatter_class=argparse.RawTextHelpFormatter
                                    , description=desc, epilog=epil)

    parser.add_argument( 'filenames', nargs='*', default=['-'],
                         help='Files to be processed. If not given, stdin is used')

    parser.add_argument( '-d', '--debug', action='store_true',
                         dest='debug',
                         help='activate debug mode -- WARNING! output is affected/spoiled!' )

    parser.add_argument( '-n', '--nohead', action='store_false',
                         dest='header',
                         help='suppress the printing of the header' )

    parser.add_argument( '-a', '--anion', type=int, nargs=1,
                         dest='anion', default=5,
                         help='set index of anion' )

    parser.add_argument( '-c', '--cation', type=int, nargs=1,
                         dest='cation', default=1,
                         help='set index of cation' )

    ## End arg parser definition
#    args = argparse.parser()
    args=parser.parse_args(sys.argv[1:])
    d = vars(args)	# adding prog in args, for unknown reasons it's not there...
    d['prog']=parser.prog

#   here the actual function doing the job is called:
    extract_angle_from_data(args)

