#!/usr/bin/env python3

import sys
import re
import subprocess
import numpy as np           # to install this lib in deb-type linux, run as root: apt-get install python3-numpy


def testforexistingcommand(command):
    p1 = subprocess.run([command], stdin=subprocess.DEVNULL, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
#    print("quiii",p1)
    err = p1.returncode
    return err==0

def failonnonexistingcommand(command):
    if not testforexistingcommand(command):
        print("Error: a needed working",command," is missing!",file=sys.stderr)
        sys.exit(1)


def run_one(load,temperature):
    """execute 1 run of lmp with the input parameters substituted
at the appropriate positions in input_scheme
"""

    label='_load_'+str(load)+'_temp_'+str(temperature)
    inputfile="input"+label
    outputfile="output"+label

    print("# starting a lmp calculation for  ",label,"... ",end = '')
    sys.stdout.flush()

# elaborate the new input file from the input_scheme :
    f = open('input_scheme', 'r')
    fw = open(inputfile, 'w')
    for line in f:
        lines=re.sub("LOAD",str(load),line)
        lines=re.sub("TEMPERATURE",str(temperature),lines)
        lines=re.sub("LABEL",label,lines)
        fw.write(lines)
    f.close()
    fw.close()

    fw = open(outputfile, 'w')
    p1 = subprocess.run(['lmp','-i',inputfile],\
                        stdout=fw, stderr=subprocess.PIPE)
    fw.close()

    if p1.returncode !=0:
        print("\nerror returned:",p1.stderr)
    
    print("done")


# the following function is only exectuted when the code is run as a script, and its purposes is to parse
# the command line and to generate a meaningful parsed args list to the actual function doing the job:
if __name__ == "__main__":
    import argparse

    desc="""repeat simulations with lmp with multiple params
"""

    epil="""\t\t\tv. 0.3 by Nicola Manini, 03/05/2020"""

    ##  Argument Parser definition
    parser = argparse.ArgumentParser( formatter_class=argparse.ArgumentDefaultsHelpFormatter
                                      , description=desc, epilog=epil)

    parser.add_argument( '--lini',
                         dest='lini', type=float, default=0.0,
                         help='initial load')
    parser.add_argument( '--lfin',
                         dest='lfin', type=float, default=10.0,
                         help='final load')
    parser.add_argument( '--lde',
                         dest='deltal', type=float, default=5.0,
                         help='load increment')
    
    parser.add_argument( '--tini',
                         dest='tini', type=float, default=0.1,
                         help='initial temperature')
    parser.add_argument( '--tfin',
                         dest='tfin', type=float, default=0.3,
                         help='final temperature')
    parser.add_argument( '--tde',
                         dest='deltat', type=float, default=0.1,
                         help='temperature increment')
    
    ## End arg parser definition
#    args = argparse.parser()
    args=parser.parse_args(sys.argv[1:])
    d = vars(args)	# adding prog in args, for unknown reasons it's not there...
    d['prog']=parser.prog

# preliminary test to veryfy if program is there at all:
    failonnonexistingcommand('lmp')

#   here the actual function doing the job is called:


    for load in np.linspace(args.lini,args.lfin,
                            int(round((args.lfin-args.lini)/args.deltal+1))):
        for temperature in np.linspace(args.tini,args.tfin,
                            int(round((args.tfin-args.tini)/args.deltat+1))):
            run_one(load,temperature)
# and here some optional print out could be added:
