#!/usr/bin/env python3

import sys
import re
import os
import os.path
import math
import cmath
import numpy as np
from scipy import optimize
import scipy.constants 


def myexp(x): # a less explosive exponential
    # even this will eventually overflow, but it does so above x=2.e50
    xmax=600.
    if x<xmax:
        if x>-xmax:
            return math.exp(x)
        else:
            return -xmax/x*math.exp(-xmax)
    else:
        return x/xmax*math.exp(xmax)
    
def fermif(bet,chempot,energ):
    return 1./(myexp(bet*(energ-chempot))+1)

def Nfunc(beta,effmass):
    return 2*math.pow(effmass/(beta*twopihbarsq),1.5)

def n(beta,mu,effmass):
    return myexp(beta*(mu-ec))*Nfunc(beta,effmass)

def p(beta,mu,effmass):
    return myexp(beta*(ev-mu))*Nfunc(beta,effmass)

def NAminus(NA,EA,beta,mu):
    return NA*fermif(beta,mu,EA-ev)

def NDplus(ND,ED,beta,mu):
    return ND*(1.-fermif(beta,mu,ec-ED))

def totalChargeDensity(mu,beta,mc,mh,NA,EA,ND,ED):
    return p(beta,mu,mh)-n(beta,mu,mc)+NDplus(ND,ED,beta,mu)-NAminus(NA,EA,beta,mu)

def semiconductor_carrier_concentration(argv):

    global ec
    global ev
    global twopihbarsq
    twopihbarsq=2*scipy.pi*scipy.constants.hbar**2/scipy.constants.elementary_charge  # 2 pi hbar^2 = 4.36136e-49, in eV instead of J

    kB=scipy.constants.Boltzmann/scipy.constants.elementary_charge # approx 0.0000861734 eV/K,  Boltzmann constant
    mE=scipy.constants.electron_mass   # approx 9.109384e-31 kg
# see https://ecee.colorado.edu/~bart/book/effmass.htm
    mc=6.**(2/3.) * (0.98*0.19*0.19)**(1/3.) *mE # conduction band effective mass (in Si, longitudinal = 0.98, transverse = 0.19, 6 equivalent minima)
    mh=0.57 *mE # valence band effective mass    (in Si, heavy hole: 0.49; light hole: 0.16)
    EA=0.045   # eV   the acceptor BINDING energy (distance from valence top)
    ED=0.045   # eV   the donor BINDING energy (distance from conduction bottom)

    NA=args.NA
    ND=args.ND
    if args.DeltaT<0:
        print("error: DeltaT has better be positive!!")
    if args.EG<=0:
        print("error: EG has better be positive!!")

    ev=0.      # eV  - the valence band maximum, arbitrary choice of 0 of energy
    ec=ev+args.EG   # eV  - the conduction band minimum

    print("#calculation of the equilibrium populations of a semiconductor with gap=",args.EG,"eV")
    print("# other bulk parameters are: m_c=",mc/mE,"*m_E, m_h=",mh/mE,"*m_E")
    print("# doping parameters are:     N_D=",ND,"m^-3, N_A=",NA,"m^-3")    
    print("#                            E_D=",ED,"eV,   E_A=",EA,"eV")    
    if args.debug: 
        print("#T [K]	mu [eV] 	  n [m^-3]      	p [m^-3] 	tot_charge [e/m^3]	iterations")
    else:
        print("#T [K]	mu [eV] 	  n [m^-3]      	p [m^-3] 	      ND+ [m^-3] 	NA- [m^-3]")

    Told=0
    if args.DeltaT==0.:  # a smart temperature list:
        dl=np.array(range(1,10))
        dlp=np.array(range(10,100))
        dlm=np.array(range(10,21))
        Temperatures=np.concatenate((0.01*dl,0.01*dlp,0.1*dlp,1.*dlp,10.*dlp,100*dlm))
    else:
        nT=int(round((args.Tmax-args.Tmin)/args.DeltaT))+1
        Temperatures=np.linspace(args.Tmin,args.Tmax,nT)
    for T in Temperatures:
        if (T-Told)/(T+Told)>0.1: # very safe bracket for mu!
            mubr1=ev
            mubr2=ec
        else:          # narrower less safe bracket to speed up the zero search:
            mubr1=mu-2*(EA+ED)
            mubr2=mu+2*(EA+ED)
        beta=1/(kB*T)
        
        mu,res=optimize.brentq(totalChargeDensity, mubr1, mubr2, args=(beta,mc,mh,NA,EA,ND,ED),xtol=2.e-16, rtol=1.e-15, maxiter=500, full_output = True)

        if args.debug: 
            print(T, mu, n(beta,mu,mc), p(beta,mu,mh), totalChargeDensity(mu,beta,mc,mh,NA,EA,ND,ED),res.iterations)
#           	Note that even at the best mu solving the equation with high precision,
#		totalChargeDensity is not precisely 0, but easily \pm 10^8:
#		this is trivially due to the finite precision of float: if more precision
#		was available, then the solution could be more accurate.
#		Note however that \pm 10^8 is entirely negligible compared to physically meaningful doping levels!
        else:
            print(T, mu, n(beta,mu,mc), p(beta,mu,mh), NDplus(ND,ED,beta,mu), NAminus(NA,EA,beta,mu))
        Told=T

#    OBSOLETE TESTS:

#        for imu in range(11):
#            mux=ev+imu*(ec-ev)/10.;
#            print("prova",T, mux, totalChargeDensity(mux,beta,mc,mh,NA,EA,ND,ED))

# tried optimize.fsolve but fails due to flat regions in totalChargeDensity: brentq works much better, once a good fork is provided:
#        mu=optimize.fsolve(totalChargeDensity, mutest, args=(beta,mc,mh,NA,EA,ND,ED), fprime=None, full_output=0, col_deriv=0, xtol=2.e-14,  maxfev=0)
#        mu=mu[0]
#        print(T, mu, n(beta,mu,mc), p(beta,mu,mh), totalChargeDensity(mu,beta,mc,mh,NA,EA,ND,ED))
#        mutest=mu
# other tests:
#        for imu in range(11):
#            mux=mu-1e-8+imu*2e-9;
#            print("provu",T, mux, totalChargeDensity(mux,beta,mc,mh,NA,EA,ND,ED))


    
# the following function is only exectuted when the code is run as a script,
# and its purposes is to parse the command line and to generate
# a meaningful parsed args list for the actual function doing the job:
if __name__ == "__main__":
    import sys
    import argparse
    commandname=sys.argv[0]

    desc="""compute the carrier concentration in a doped semiconductor
in the 2-band model, given its doping concentrations ND and NA
for details, see Hook Hall "Solid State Physics", Chapter 5
OUTPUT:  T mu n p ND- NA+"""

    epil="""\t\t\tv. 2.1 by Nicola Manini, 29/04/2020"""

    ##  Argument Parser definition:
    parser = argparse.ArgumentParser( formatter_class=argparse.ArgumentDefaultsHelpFormatter
                                    , description=desc, epilog=epil)

    parser.add_argument( 'filenames', nargs='*', default=['-'],
                         help='Files to be processed. If not given, stdin is used')

    parser.add_argument( '-d', '--debug', action='store_true',
                         dest='debug', 
                         help='activate debug mode -- WARNING! output is affected/spoiled!' )

    parser.add_argument( '--NA',
                         dest='NA', type=float, default=1.e17,
                         help='concentration of acceptors     per cube meter')

    parser.add_argument( '--ND',
                         dest='ND', type=float, default=1.e20,
                         help='concentration of donors     per cube meter')

    parser.add_argument( '--EG',
                         dest='EG', type=float, default=1.1,
                         help='gap energy  [eV]')

    parser.add_argument( '--Tmin',
                         dest='Tmin', type=float, default=1.,
                         help='lowest temperature  [Kelvin]')

    parser.add_argument( '--Tmax',
                         dest='Tmax', type=float, default=1000.,
                         help='highest temperature  [Kelvin]')

    parser.add_argument( '--DeltaT',
                         dest='DeltaT', type=float, default=0.,
                         help='temperature increment  [Kelvin]')

    ## End arg parser definition
#    args = argparse.parser()
    args=parser.parse_args(sys.argv[1:])
    d = vars(args)	# adding prog in args, for unknown reasons it's not there...
    d['prog']=parser.prog
#   print('outside, in parser:',args)

#   here the actual function doing the job is called:
    semiconductor_carrier_concentration(args)

