#!/usr/bin/env python3

import numpy as np # to install this lib in deb-type linux, run as root: apt-get install python3-numpy
import xyz_utils
from collections import namedtuple

def xyz_join(filenames):
    """open and read one all filenames frame by frame and merge them
    returns a full xyz object with all combined frames"""

    if len(filenames)<1:
        print("error: need at least 1 file!!!", file=sys.stderr)
        return None
    
    flist=[]
    for filen in filenames: # open them all!
        if filen=="-":
            f=sys.stdin
        else:
            f = open(filen, 'r')
        flist.append(f)

    allfr=[]
    repeat=True
    while repeat: # loop on the smapshots of all files
        atoms=[]
        coords = np.empty( shape=(0, 3) )
        ntot=0
        comment="#joined"
        for f in flist:
            fr=xyz_utils.xyz_read_one_frame(f) # read one frame from each file
            if fr.atoms != None:
                comment+=" "+fr.comment
                atoms+=fr.atoms
                coords=np.concatenate((coords,fr.coords),axis=0)

        if len(atoms)>0:
            tfr=namedtuple("xyzdata", ["comment", "atoms", "coords"])(comment, atoms, coords)
            allfr.append(tfr)
        else:
            repeat=False

    return allfr    
        

# the following function is only exectuted when this code is run as a script, and its purposes is to parse
# the command line and to generate a meaningful parsed args list to the actual function doing the job:
if __name__ == "__main__":
    import sys
    import argparse
    commandname=sys.argv[0]

    desc="""Join several xyz files into one. 
Even with different numbers of frames and numbers of atoms in each frame!
INPUT: various xyz files
OUTPUT: a xyz file """

    epil="""                    v2.2 by N. Manini, 23.04.2020"""

##  Argument Parser definition:
    parser = argparse.ArgumentParser( formatter_class=argparse.ArgumentDefaultsHelpFormatter
                                    , description=desc, epilog=epil)

    parser.add_argument( 'filenames', nargs='*', default=['-'],
                         help='Files to be processed. If not given, stdin is used')

## End arg parser definition
    args=parser.parse_args(sys.argv[1:])
    d = vars(args)	# adding prog in args, for unknown reasons it's not there...
    d['prog']=parser.prog
#   here the actual function doing the job is called:
    a=xyz_join(args.filenames)
#   final printout:
    xyz_utils.xyz_write_entire_file(sys.stdout, a)
