#!/usr/bin/env python3

import sys
import numpy as np
import xyz_utils
from collections import namedtuple


def frame_copy(fr):
    """return a copy of the xyz frame fr in input"""
    comment=fr.comment
    atoms=fr.atoms
    coords=fr.coords
    return namedtuple("xyzdata", ["comment", "atoms", "coords"])(comment, atoms, coords)

def xyz_select_frame(args):

    if args.last: # last prevails over nframe
        args.nframe=0

    for filen in args.filenames:
        if filen=="-":
            f=sys.stdin
        else:
            f = open(filen, 'r')

        nf=0
        lif=[]
        while True: # loop on the smapshots of all files
            nf+=1
            fr=xyz_utils.xyz_read_one_frame(f)
            if fr.atoms == None:
                break
            else:
                frcopy=frame_copy(fr)
                if nf==args.nframe:
                    lif.append(frcopy)
                    break
        if args.last:
            lif.append(frcopy)
        return lif


# the following function is only exectuted when this code is run as a script, and its purposes is to parse
# the command line and to generate a meaningful parsed args list to the actual function doing the job:
if __name__ == "__main__":
    import argparse
    commandname=sys.argv[0]

    desc="""Select a specified frame from a xyz file
INPUT: various xyz files
OUTPUT: a xyz file with just one frame from each of the give files"""

    epil="""                      v0.1 by N. Manini, 6.05.2020"""

##  Argument Parser definition:
    parser = argparse.ArgumentParser( formatter_class=argparse.ArgumentDefaultsHelpFormatter
                                    , description=desc, epilog=epil)

    parser.add_argument( 'filenames', nargs='*', default=['-'],
                         help='Files to be processed. If not given, stdin is used')

    parser.add_argument( '-n',
                         dest='nframe', type=int, default=1,
                         help='the sequential number 1, 2, 3... of the frame to be selected')

    parser.add_argument( '-l',
                         dest='last', action='store_true',
                         help='return the last frame')

## End arg parser definition
    args=parser.parse_args(sys.argv[1:])
    d = vars(args)	# adding prog in args, for unknown reasons it's not there...
    d['prog']=parser.prog
#   here the actual function doing the job is called:
    lif=xyz_select_frame(args)
    for fr in lif:
        xyz_utils.xyz_write_one_frame(sys.stdout, fr)

        
