/*
This software was developed by Alessio Del Monte and Nicola Manini. It is not
subject to copyright protection and is in the public domain: permission is
granted to any individual or institution to use, copy, modify or redistribute
it. The authors make no guarantees about this software and assume no
responsibility for its use by other parties.

Whoever makes use of it is asked to cite "A. Del Monte, N. Manini, L.G.
Molinari, and G.P. Brivio, Mol. Phys. 103, 689 (2005)" and the URL
http://materia.fisica.unimi.it/manini/ivr.html.

This license statement should be provided in derived software.
*/

#include <iostream>
#include <fstream> 
#include <string>
#include <sstream>
#include <iomanip>

#include <vector>
#include <valarray>
#include <map>

#include <cassert>
#include <functional> // std::greater
#include <complex>


#include "global.hpp" // OVERT_MAX and typedef

#include "lib_algorithms.hpp" 
#include "lib_vector.hpp" 

#include "tensor.hpp"
#include "state.hpp"
#include "tier.hpp"
#include "lib_symmat.hpp"


#define DEFAULT_ENERGY_WINDOW 1e99
#define DEFAULT_TIER_SIZE 1000

void read_configuration(std::ifstream& config_file);

using namespace std;

namespace prm
{

  double window_inf = 0.0, window_sup =  10.0; 
  
  double* energy_inf; double*  energy_sup; // energy window

  double epsilon = 0.1, delta = 0.4;
 
  unsigned tiers_number = 3;

  unsigned size_of_tier(unsigned tier);

  vector<unsigned> tier_size;
 
  unsigned max_generated_states = 2000;
   
  vector<string>  bright_vector; 
  string coefficients = "phi3.dat"; // list of files with potential coefficient
  string frequencies = ""; // file with normal mode in the form "1 1232.3"
}


  

namespace opt
{

  bool show_tiers = false ;
  
  bool show_matrices = false;
  bool show_phi = false;
  bool show_weights = false;


  bool avoid_green = false;
  bool dynamic_resize = false;
  bool avoid_alarm = false;

  bool zpe = true;
  bool banal_inversion = false;

  bool square_ccs = true; // doing square of c.c.s.
  bool sum_ccs = true; // adding c.c.s. of states found more than once
  // bool square_sum = true;
}



map<unsigned, unsigned, std::greater<unsigned> > tier_size_map;

int main(int argc, char* argv[])
{

  if (argc  <= 1 ) {
 	 cerr << "ERROR: Configuration file pathname not specified at command line." << endl;
	 return EXIT_FAILURE;
  }
  else if (argc > 2 ) {
	 cerr << "ERROR: One single parameter (the configuration file) should be passed at command line." << endl;
	 return EXIT_FAILURE;
  }

  ccout <<"***** SETTINGS AND POTENTIAL COEFFICIENTS FROM INPUT ***** "
	<<newlc<<'\n';

  ifstream config_file;
  config_file.open(argv[1]);
  if(!config_file.is_open()) {
	 	 cerr << "ERROR: Cannot open configuration file \"" << argv[1] << "\"" << endl;
		 return EXIT_FAILURE;
  }
  try {
	  read_configuration(config_file); // parameters and options from input
  }
  catch(std::exception& exc) {
	  cerr << "ERROR: Exception while reading configuration - " << exc.what() << endl;
	  config_file.close();
	  return EXIT_FAILURE;
  }
  config_file.close();

  //cout <<prm::coefficients;
  
  try {
	  State::initialize(prm::coefficients.c_str(), prm::frequencies.c_str());
  }
  catch(std::exception& exc) {
	  cerr << "ERROR: Exception while reading input parameters - " << exc.what() << endl;
	  config_file.close();
	  return EXIT_FAILURE;
  }

  if(opt::show_phi) {
    ccout <<"Potential coefficients from input files"<<"\n";
    ccout << State::APESurface<<"\n";
  }
  

  // Renormalization
  //
  ccout << "Number of normal modes: " << State::APESurface.count_modes()<<"\n";
 
  State::APESurface.Taylor_renormalization();
  // ccout <<"Renormalized: "<<"\n"<<State::APESurface<<"\n";
  if(opt::sum_ccs)
    State::APESurface.Schwartz_renormalization();
  // see tier.hpp

  State::APESurface.quantum_renormalization();
 

#ifdef GREEN_DBG
  ccout <<"Quantum renormalized coefficients: "<<newlc<<State::APESurface<<"\n";
#endif
  
  //***** Tier basis generation
  //
  
  TierBasis basis(prm::bright_vector); // tier basis, with tier 0 of bright
 

  ccout<<'\n';

  State fundamental_state("");

  ccout<<"Harmonic energy of unperturbed 0-phonon state is "
        << fundamental_state.energy()<<"\n";

  ccout<<"List of states in the Tier 0 (bright state)\n";
  basis.front().print_verbose();

  
  cerr<<"Building the basis of zero-order states ... "<<endl;

  for (unsigned int i = 1; i <= prm::tiers_number; i++) 
    { // generation of tiers
      cerr<<"- building tier n." <<i<< " with maximum size "
	  <<prm::size_of_tier(i)<<endl;
      basis.next_tier(prm::size_of_tier(i)); //[[ max_size 17.4.1.8
    }

  ccout <<"\n";
  if(opt::show_tiers) {
    ccout <<"***** BASIS ***** "<<newlc<<'\n';
    ccout <<"Basis of zero-order states: \n"<<basis;
  }
    ccout <<"\n";

 
  
  cerr<<"Generating the Hamiltonian matrix ..."<<endl;

 if(!opt::sum_ccs)
    State::APESurface.Schwartz_renormalization();

  SBTMatrix HMatrix(basis);
  
  
  // ccout <<"Max diagonal capacity: " << HMatrix.max_diagonal_capacity()
  // 	<<newlc<<"Max over-diagonal capacity: " 
  //    << HMatrix.max_overdiagonal_capacity()<<"\n";

  if (opt::avoid_green) { 
      std::cerr<<"Skipping Green function calculation"
	  << (opt::avoid_alarm? "" : "\a") <<"\n"; 
      return 0;
  }   
  ccout<<endl; 

  //***** Loop on frequencies
  //
 cerr<<"Calculating the Green function ..."<<endl;

 
  ccout <<"***** GREEN FUNCTION ***** "<<newlc<<"\n";
  // Matrices for the block inversion procedure
  //

  TNT::SyMatrix<Fortran_complex > GM; // Green matrix block <k | G | k> 
  TNT::Matrix<Fortran_complex > Mtemp; // working matrix
  
  if(!opt::dynamic_resize) { // reserving space enough for static
			   // resizing
     GM.reserve(HMatrix.max_diagonal_capacity());
     Mtemp.reserve(HMatrix.max_overdiagonal_capacity()); 
  }

  for(double omega =  prm::window_inf; omega <= prm::window_sup; 
     omega += prm::delta) // moves along frequencies (x-axis)
    {
      Fortran_complex z_ (omega, prm::epsilon); 
     
      // Iterative procedure
      //
      SBTMatrix::reverse_iterator pblock; // Sec. 16.3.2, 19.2.5
            
      for(pblock = HMatrix.rbegin(); pblock != HMatrix.rend();
	  pblock++) {

	if (pblock == HMatrix.rbegin()) { // the last first block
	  unsigned dim = pblock->diag().num_rows(); 
	  if(opt::dynamic_resize) // resizing the GM
	    GM.newsize(dim, dim);	    
	  else
	    GM.static_newsize(dim, dim); 

	
	  GM = Fortran_complex();// Note: all elements are put to zero
				 // (indeed this is already true in
				 // the case of non-static resize)
	}
	else { // -=  <k-1|V|k> <k|G|k> t<k-1|V|k> 
	  sparse::matmult_t(Mtemp, GM,  pblock->overdiag(), !opt::dynamic_resize);
	  sparse::matmult(GM, pblock->overdiag(), 
			  Mtemp, std::minus<Fortran_complex>(), 
			  !opt::dynamic_resize); // in GM new matrix
	}
	assert(GM.num_rows() == GM.num_cols());
	assert(GM.num_rows() == pblock->diag().num_rows());
	
	GM.add_on_diagonal(z_); // += z_ Id
	
	sparse::add(GM, pblock->diag(), 
		    std::minus<Fortran_complex>(),// -= <k-1 | H | k-1>
		    true // unnecessary to complete lower triangle,
			 // since invert() works on the upper triange
			 // only
		    ); 
	// NOTE: only the upper triangle is added
#ifndef ONLY_BANAL_INVERSION  
	if(!opt::banal_inversion) {
	  // cmat::double_complex_symmetric_indefinite_inversion(GM);
	  GM.invert();
	}
	else 
#endif
	  GM.banally_invert();
	
      } // loop: iteration on list of blocks
      
      cout << setprecision(16)<<omega;
      for (int i = 0; i < GM.num_rows(); i++)
	for (int j = 0; j < GM.num_rows(); j++) {
	  cout << setprecision(16)<<'\t'<< -1 *imag(GM[i][j]) <<' '
	       <<real(GM[i][j])<<"  ";
	}
      cout << "\n"; 

    } // loop on frequencies

  cerr<<"Execution completed with success" << (opt::avoid_alarm? "" : "\a") << endl;
  return EXIT_SUCCESS;
 } // main


void read_configuration(ifstream& config_file) {

  string line; // a single line from input
  while (getline(config_file, line))
    { 
      istringstream iss(line);

      string token; // temporary string
      const string input_sign = ">>"; // input of parameters
      const string option_sign = "$"; // flags assignment

      string got_sign;
      iss >> token; iss >> got_sign; 

      //***** Input of parameters and file names
      //

      if (token.length() > 0  && // skipping white lines
	  got_sign == input_sign )   
	{
	 if (token == "window") 
	    {
	      iss >> prm::window_inf >> prm::window_sup; 
	      assert (prm::window_inf <= prm::window_sup);
	    }
	 else if (token == "epsilon") iss >> prm::epsilon; 
	 else if (token == "delta") iss >> prm::delta; 
	 else if (token == "tiers") iss >> prm::tiers_number;
	 //	 else if (token == "energy") 
	  else if (token == "tier_size")
	    { // only after input of number of tier
	      int tier, size;
	      iss >> tier>> size;

	      if(tier <= 0 || size <= 0) {
		cerr<<"ERROR: negative values while setting maximum tier size"
		    <<endl;
		exit(1);
	      }
 
	      if(!tier_size_map.insert(make_pair(unsigned(tier),
						 unsigned(size))).second){
		cerr<<"WARNING: ignored multiple setting of size of tier "
		    <<tier<<endl;
	      } // insert(): 17.4.1.7
	    }

	  else if (token == "maxgenerated")
	    iss >> prm::max_generated_states; 
	 
	  else if (token == "bright") 
	    {
	      string linerest; getline(iss, linerest);  // rest of line
	      prm::bright_vector.push_back(linerest);  
	    }
	
	  // Names of files with parameters
	  //

	 else if (token == "coefficients")
	    getline(iss, prm::coefficients); // 	
	 
	 else if (token == "frequencies") {
	   getline(iss, prm::frequencies);
	 }
	 
	} // got_sign == input_sign

      //***** Flags assignments
      //

      else if (token.length() > 0  && got_sign == option_sign)
	{ 
	  if (token == "turnoff") { 
	      string linerest; getline(iss, linerest); 
	      istringstream reststream (linerest); // apro l'oggetto di iss
	      string word;
	      while (reststream >> word) {
		  if (word == "spectrum") opt::avoid_green = true; 
		  else if (word == "zpe") opt::zpe = false;
		  else if (word == "static") opt::dynamic_resize = true; 
		  else if (word == "square") opt::square_ccs = false;
		   else if (word == "sum") opt::sum_ccs = false;
		  else if (word == "inversion") opt::banal_inversion  = true;
		  else if (word == "alarm") opt::avoid_alarm = true;
	      }
	    }
	  else if (token == "show") { 
	      string linerest; getline(iss, linerest); 
	      istringstream reststream (linerest); 
	      string word;
	      while (reststream >> word) {
		  if (word == "tiers") opt::show_tiers = true;
		  else if (word == "matrices") opt::show_matrices = true; 
		  else if (word == "coefficients") opt::show_phi = true; 
		  else if (word == "weights") opt::show_weights = true; 
		}
	    }
	  
	} // got_sign == option_sign

    } // while(getline)

}

unsigned prm::size_of_tier(unsigned tier) {

  assert(tier > 0);
  map<unsigned, unsigned>::const_iterator lb = 
    tier_size_map.lower_bound(tier); //  tier_size_map in decreasing order
  
  unsigned size;
  if(lb != tier_size_map.end()) {
    assert(lb->first > 0); // not to be set the size of tier 0
    size = lb->second;
  }
  else // 
    size =  DEFAULT_TIER_SIZE;

  return size;
}

