/*
This software was developed by Alessio Del Monte and Nicola Manini. It is not
subject to copyright protection and is in the public domain: permission is
granted to any individual or institution to use, copy, modify or redistribute
it. The authors make no guarantees about this software and assume no
responsibility for its use by other parties.

Whoever makes use of it is asked to cite "A. Del Monte, N. Manini, L.G.
Molinari, and G.P. Brivio, Mol. Phys. 103, 689 (2005)" and the URL
http://materia.fisica.unimi.it/manini/ivr.html.

This license statement should be provided in derived software.
*/


/* SPARSE MATRIX CLASS

  Sparse matrices in packed storage, following the row-indexed scheme
  described in Numerical Recipes. Both the public interface and the
  internal implementation are kept as much as possible similar to the
  definition of the class Matrix in the Template Numerical Toolkit (TNT)
  library (see "tnt_cmat_modified.h").
 
  Note that if the matrix is not sparse enough, more memory is needed
  than using ordinary storage.

*/

#include <cstdlib>
#include <cassert>
#include <iostream>
#include <algorithm> // lower_bound()
#include <functional> // plus

#include "tnt_subscript.h" // To have defined TNT_SUBSCRIPT_TYPE

#define SPARSE_SUBSCRIPT_TYPE TNT_SUBSCRIPT_TYPE
// NOTE: the use of a different subscript type than that employed in
// TNT Matrix library, would cause lost of efficiency in linear
// algebra algorithms due to the implicit cast.

#include "tnt_cmat_modified.h"

// NOTE: For debugging define macro SPARSE_CHECK for (time expansive) checks.

namespace sparse
{
 
 typedef SPARSE_SUBSCRIPT_TYPE Subscript; 

  template <class T>
  class Matrix_base // do not know TNT Matrix class for ordinary matrices
  {
   public:

    typedef Subscript   size_type; 
    typedef         T   value_type;
    typedef         T   element_type;
    typedef         T*  pointer;
    typedef         T*  iterator; 
    typedef         T&  reference;
    typedef const   T*  const_iterator;
    typedef const   T&  const_reference;

  protected:

    Subscript m_ ; // rows number of the original matrix
    Subscript n_ ; // columns 

    T threshold_; // threshold distinguishind elements to be catched

    Subscript capacity_; // compressed size: length of both col_[] and v_[];

    Subscript* row_; // partition of col_ and v_ (position of the
                     // first element of each row)

    Subscript* col_; // entries in col_ contain the column number of
                     // the corresponding element in v_

    T* v_; // entries in v_ contain the elements of the original
           // matrix with absolute value grater than the threshold

    // Note that row[m_] == capacity


    // Protected functions 
    //
    void initialize(Subscript M, Subscript N, const T* v, T thresh = T()) 
    {
      m_ = M; n_ = N;  threshold_ = thresh; 
    

      capacity_ = compressed_size(v);
      initialize(); // allocation of arrays
      
      compress(v);
    } // initialize(...)

    void initialize()
    {
    
      row_ = new Subscript[m_ + 1]; 
      // '+1' in order to know the capacity_ of the last row
      v_ = new T[capacity_];
      col_  = new Subscript[capacity_];

      assert(v_  != NULL); // alloc was successful
      assert(row_  != NULL);
      assert(col_ != NULL);
    } // initialize()

   
    void compress(const T* vec) // virtual??
    {
     
      const T* p = vec; // moves on the uncompressed row-oriented matrix
      Subscript k = 0; // moves on v_ and col_

      for (Subscript i = 0; i< m_; i++) 
    {
      row_[i] = k; // first element of the i-th row in v_ and col_
      p += jump(i);
      // In the case of symmetric matrices, the jump p += i is the
      // same that considering zero all under-diagonal elements.

      for (Subscript j = jump(i); j < n_; j++)
        {
    #ifdef SPARSE_CHECK
          assert (k <= capacity_);
    #endif
          if (std::abs(*p) > threshold_) // abs() in <cmath>, Sec. 22.3
        {
          v_[k] = *p;
          col_[k] = j;
          k++;
        }
          p++;
        }

    } // loop on rows

      row_[m_ ] = capacity_; 
    } // compress()


    Subscript compressed_size (const T* vec) 
    { 
     
      Subscript num = 0;

      const T* p = vec ; 

      for (int i = 0; i< m_; i++)
    {
      p += jump(i); // see above
      for (int j = jump(i); j< n_; j++)
        {
          if (std::abs(*p) > threshold_)
        num++;
          p++;
        }
    }
      return num; 
     
    } // compressed_size()
   
    void copy(const sparse::Matrix_base<T>&  A)
    {
      Subscript i;
      for (i=0; i< capacity_; i++) {
        v_[i] = A.v_[i];
        col_[i] = A.col_[i];
      }

      for(i = 0; i < m_ +1; i++)
        row_[i] =   A.row_[i];
    }  // copy()

    void destroy()
    {     
      /* do nothing, if no memory has been previously allocated */
      if (v_ == NULL) return ; 
    
      /* if we are here, then matrix was previously allocated */
      
      assert(col_ != NULL); 
      assert (row_ != NULL); 

      delete [] (v_);   // %% inutili parentesi ? 
      delete [] (col_); 
      delete [] (row_);
    
      capacity_ = 0; 
    } // destroy()
    
    virtual Subscript jump(Subscript i){ return Subscript(); }
    // virtual in order to have polymorphic behavior (see SyMatrix)

    

  public:

    Subscript lbound() const { return 1;} 
  
    // Constructors and assignment
    //
  
    Matrix_base() : m_(0), n_(0), threshold_(0), capacity_(0), row_(0), 
		    col_(0), v_(0) {} 
  
 
    sparse::Matrix_base<T>& operator=(const sparse::Matrix_base<T> &A)
    {
        if (v_ == A.v_) {
            std::cerr <<"WARNING: trying assignment of a sparse matrix to itself" << std::endl;
            return *this;
        }
        else
        {
            m_ =A.m_; n_ = A.n_; capacity_ = A.capacity_;
            threshold_ = A.threshold_;

            destroy();
            initialize();
            copy(A);
        }
    
      return *this;
    }  
  
    virtual ~Matrix_base()
    {
      destroy();
    }

    // Access functions
    //

    Subscript num_rows() const { return m_; }
    Subscript num_cols() const { return n_; }
    Subscript capacity() const { return capacity_; }

    T threshold() const { return threshold_; }
    
 
    inline Subscript row(Subscript i ) const 
    {
#ifdef SPARSE_CHECK
      assert(i >= 0); assert(i <= m_) ;
#endif
      return row_[i]; 
    }
    // qualifier const for add() (sparse matrix is const)


    inline T val(Subscript k ) const
    {
#ifdef SPARSE_CHECK
      assert(0<= k); assert(k < capacity_) ;
#endif
      return v_[k]; 
    }
  
    inline Subscript col(Subscript k ) const  // %% old name_ j_
    {
#ifdef SPARSE_CHECK
      assert(0<= k); assert(k < capacity_) ;
#endif
      return col_[k]; 
    }

    // returns the position on v_ of entry (i,j) if it exists, else
    // the position of first element of the (i+1)th-row, i.e. row_[i+1]
    inline Subscript col_lower_bound(Subscript i, Subscript j) const {
#ifdef SPARSE_CHECK
      assert(i >= 0); assert(i < m_); assert(j>= 0); 
      assert(j<n_); //[[ ev. <= to take account of extreme of row_
#endif
      Subscript* first = &col_[row_[i]];
      Subscript* last = &col_[row_[i + 1]];
      Subscript* p = std::lower_bound(first, last, j); 
      // lower_bound(), Stroustrup Sec. 18.7.2: if the search of the
      // element failed, lower_bound() returns the iterator to the
      // first element (with key) greater than it or last if not any

      return(*p == j ? // binary search succeeded?
                  p - col_ : row_[i + 1]); // row_[i + 1] == last - col_
    }


    inline T operator() (Subscript i, Subscript j) const
    {
#ifdef SPARSE_CHECK
      assert(i >= 0); assert(i < m_); assert(j>= 0); 
      assert(j<n_); 
#endif
      Subscript pos = col_lower_bound(i,j);
      
      return (pos != row_[i+1] ? v_[pos] : T() );
    }
    // Note 1: reference return would be impossible for elements rejected 

    // Note 2: useful but not efficient: do not use in linear-algebra
    // algorithms
  
    // Utils
    //

    inline double compression_factor() {
      return double(capacity_) / ( m_ * n_); 
    }

    inline unsigned size_of() { // memory occupied (in byte)
      return  sizeof(Subscript) * (m_+ capacity_) + sizeof(T) * capacity_;
    }
    // sum size of row_, col and v_

    inline double memory_compression_factor() {
      return double(size_of()) / (sizeof(T) * m_ * n_);
    }

    void inner_print() { // for development
      std::cout <<"Sparse matrix "<<m_ <<'*' <<n_ << " ("<<capacity_<<")\n";
      
      Subscript i;
      std::cout <<"- values in v_:\n" ;
      for(i = 0; i < capacity_ ; i++)
	std::cout << v_[i]<<'\t';
      std::cout <<"\n- indexes of columns in col_:\n" ;
      for(i = 0; i < capacity_ ; i++)
	std::cout << col_[i]<<'\t';
      std::cout <<"\n- partitioning of v_ in row_:\n" ;
      for(i = 0; i < m_; i++) 
	std::cout << i <<':'<<row_[i]<<' ';
    }
 
   }; // class sparse::Matrix_base


 template<class T>
 class Matrix: public Matrix_base<T>
  { // added to  class Matrix_base constructor from TNT::Matrix

  public:
    Matrix() : Matrix_base<T>() {}
    Matrix(const TNT::Matrix<T>& A, T aThresh = T()) :
      Matrix_base<T>() {
      this->initialize(A.num_rows(), A.num_cols(), &A[0][0], aThresh); 
    }

  
  }; // class sparse::Matrix



template<class T>
 class SyMatrix: public Matrix_base<T>
  { // added to  class Matrix_base constructor from TNT::Matrix
    
  public:
    SyMatrix() : Matrix_base<T>() {}
    SyMatrix(const TNT::Matrix<T>& A, T aThresh = T()) :
      Matrix_base<T>() {
      this->initialize(A.num_rows(), A.num_cols(), &A[0][0], aThresh);
    }

  protected:

    Subscript jump(Subscript i){ return i; }  
    // virtual meber function

  }; // class sparse::SyMatrix


  
 
  //***** Utils functions
  //
  //%% ev. private: then pointer to member function
  template<class T>
  inline T return_id(T i) { return i; } // identity 
  
  template<class T>
  inline T return_0(T i) { return T(); } 
  

  //****************************[ Output ]****************************
  
  template <class T>
  std::ostream& operator<<(std::ostream &s, const sparse::Matrix<T> &A)
  {
    Subscript M=A.num_rows();
    Subscript N=A.num_cols();

    s << M << " " << N;
    if( A.threshold() == T())
      s <<'\n';
    else 
      s<<" (" << A.threshold()<< ")\n";

    for (Subscript i=0; i<M; i++)
      { 
        for (Subscript j=0; j<N; j++)
            s << A(i, j) << " ";
        s << "\n";
      }
    return s;
  }


  template <class T>
  std::ostream& operator<<(std::ostream &s, const sparse::SyMatrix<T> &A)
  {
    Subscript M=A.num_rows();
    Subscript N=A.num_cols();

    s << M << " " << N;
    if( A.threshold() == T())
      s <<'\n';
    else 
      s<<" (" << A.threshold()<< ")\n";

    for (Subscript i=0; i<M; i++)
      { 
        for (Subscript j=0; j<N; j++)
            s << (i <= j ? A(i, j) : A(j,i)) << " ";
        s << "\n";
      }
    return s;
  }

  // *******************[ Basic matrix algorithms ]***********************


  template <class T, class R, class BinOp> // A += B or A -= B 
  void add(TNT::Matrix<T> &A, sparse::Matrix_base<R> & B, 
	   BinOp op, // sign + or -
	   bool fill_lower = true) 
    // turn fill_lower to false if you do not need to sum entries of
    // the lower triangle in the case matrix B is symmetric

    //[[const: "conversion casts away constness": Sec. 15.4.2.1 ?
  {
    assert(A.num_rows() == B.num_rows());
    assert(A.num_cols()== B.num_cols());

    Subscript M = A.num_rows();
  
    for (Subscript i = 0; i < M; i++)
      for (Subscript k = B.row(i); k < B.row(i+1); k++) {
	A[i][B.col(k)]  = op(A[i][B.col(k)], B.val(k) ); 
      }
    
   
    /*try {*/
      sparse::SyMatrix<R>& SyB = dynamic_cast<sparse::SyMatrix<R>& >(B);

      // if we are here, then sparse matrix B is symmetric, see
      // dynamic_cast, sec. 15.4.1.1 (see Sec. 12.2.6, 12.3 for
      // datails on polymorphism)

      if(fill_lower) {	
        for (Subscript i = 0; i < M - 1; i++) // M-1 to exclude last element
          for (Subscript k = SyB.col_lower_bound(i, i+1); k < SyB.row(i+1); k++)
            {
              A[B.col(k)][i]  = op(A[SyB.col(k)][i], SyB.val(k) );
            }
      }// if(fill_lower)
    /*}
    catch (std::bad_cast) { // do nothing
#ifdef GREEN_CHECK
      std::cerr<<"WARNING: add() should receive symmetric sparse matrix "
	"in program Green " <<endl;
#endif
    } */
   

  } // add()
  // Note: do not yet implemented the case 

  
  template <class T, class R, class BinOp> // C = A * B (A sparse)
  //%%template <class T, class R, class BinOp = std::plus<T> > 
  //%% default arguments of template: Sec. 13.4.1 ???%%%; plus: Sec. 18.4.3
  inline void matmult(TNT::Matrix<T>& C, const sparse::Matrix<R>  &A,
          const TNT::Matrix<T> &B,
          BinOp op, // C = -(A * B) with op = minus<T>
          bool static_resize = false) //
  { 
    assert(A.num_cols() == B.num_rows());

    Subscript M = A.num_rows();
    Subscript N = B.num_cols();

    if(static_resize)
      C.static_newsize(M,N);
    else
      C.newsize(M,N);
    
    T sum;

    for (Subscript i = 0; i<M; i++)
      for (Subscript j = 0; j<N; j++)
        {
          sum = 0 ;

          for (Subscript k = A.row(i); k < A.row(i+1); k++)
            {
              sum = op(sum,  A.val(k) * B[A.col(k)][j]);
            }
          C[i][j] = sum;
        }
    return;
  } // matmult()

  template <class T, class R> // C = A * tB (B sparse)
  inline void matmult_t (TNT::Matrix<T>&  C,  const TNT::Matrix<T>&  A,
        const sparse::Matrix<R> & B,
        bool static_resize = false)
  {
    assert(A.num_cols() == B.num_cols());

    Subscript M = A.num_rows(); 
    Subscript N = B.num_rows(); 

    if(static_resize)
      C.static_newsize(M,N);
    else
      C.newsize(M,N);
 
    T sum; 

    for (Subscript i = 0; i<M; i++)
      for (Subscript j = 0; j<N; j++)
        {
          sum = 0 ;

          for (Subscript k = B.row(j); k < B.row(j+1); k++)
            {
              sum +=  A[i][B.col(k)] * B.val(k) ;

            }
          C[i][j] = sum;
	}
    return;
  } // matmult_t()

}// namespace sparse
