/*
This software was developed by Alessio Del Monte and Nicola Manini. It is not
subject to copyright protection and is in the public domain: permission is
granted to any individual or institution to use, copy, modify or redistribute
it. The authors make no guarantees about this software and assume no
responsibility for its use by other parties.

Whoever makes use of it is asked to cite "A. Del Monte, N. Manini, L.G.
Molinari, and G.P. Brivio, Mol. Phys. 103, 689 (2005)" and the URL
http://materia.fisica.unimi.it/manini/ivr.html.

This license statement should be provided in derived software.
*/

#include<typeinfo>

#include "tnt_094_fortran_modified.h"

// Fortran subroutines of Lapack linear-algebra library
//
#define F77_DSYTRF dsytrf_ // see below
#define F77_DSYTRI dsytri_

#define F77_ZSYTRF zsytrf_
#define F77_ZSYTRI zsytri_


extern "C" { // extern "C": see Sec. 9.2.4

  // Double precision complex symmetric indefinite matrix 
  //
  void F77_DSYTRF(cfch_ uplo, cfi_ N, fda_ A, cfi_ lda, fia_ ipiv, fda_  work, 
		  cfi_ lwork, cfi_ info ); 
  void F77_DSYTRI(cfch_ uplo, cfi_ N, fda_ A, cfi_ lda, fia_ ipiv, fda_ work, 
		  cfi_ info ); 
  // Compute the inverse of a matrix using the LU factorization
  // computed by DSYTRF

  // Double precision complex symmetric indefinite matrix 
  //
  void F77_ZSYTRF(cfch_ uplo, cfi_ N, fca_ A, cfi_ lda, fia_ ipiv, fca_ work, 
		  cfi_ lwork, cfi_ info ); 
  void F77_ZSYTRI(cfch_ uplo, cfi_ N, fca_ A, cfi_ lda, fia_ ipiv, fca_ work, 
		  cfi_ info ); 

} // extern "C"

#include <stdexcept>

#include "tnt_cmat_modified.h"

namespace TNT {
  
  
  template <class T> // General squared M*M matrix 
  class SqMatrix : public TNT::Matrix<T>
  {

  public:
    SqMatrix() : TNT::Matrix<T>() {}

    SqMatrix(const Matrix<T> &A) : Matrix<T>(A) { assert(TNT::Matrix<T>::m_ == this->n_); }
  
    SqMatrix(Subscript M, Subscript N, const T& value = T()) :
      Matrix<T>(M,N, value)  { assert(this->m_ == this->n_); }
   
    SqMatrix(Subscript M, Subscript N, const T* v):  
      Matrix<T>(M,N, v) { assert(this->m_ == this->n_); }

    SqMatrix(Subscript M, Subscript N, const char *s) : 
      Matrix<T>(M,N,s)  { assert(this->m_ == this->n_); }


    SqMatrix<T>& operator=(const T& scalar)
    { 
      this->set(scalar); 
      return *this;
    }
    // operator= must be redefined, since assignment operators are not
    // inherited, Sec. 12.2.3 (nor it is sufficient to declare it
    // `virtual')


    // New features added to original TNT linear algebra methods
    // 

    void add_on_diagonal(T val )
    {
        for (Subscript i = 0; i < this->m_ ; i++) // added  z * Id
            this->row_[i][i] += val;
    }
    
    void invert()
    { // matrix inversion
      throw std::domain_error("Cannot use Lapack inversion on a generic squared matrix in row-oriented storage");
    }

    //Inversion of diagonal elements, putting non-diagonal ones to zero
    //
    void banally_invert()
    {
        for (Subscript i = 0; i < this->m_; i++) {
            for (Subscript j = 0; j < this->n_; j++) {
                if(i==j) {
                    this->row_[i][j] = pow(this->row_[i][j],-1); // pow for complex numbers: see Sec. 22.5
                    // if(typeid(T) != typeid(Fortran_complex)) // lost of performance
                    //assert(this->row_[i][j] != T()); // non-zero for real matrix
                }
                else {
                    this->row_[i][j]=0;
                }
            }
        }
    } // banally_invert()

    bool isSymmetric()
    {
#ifdef GREEN_CHECK 
      std::cerr<<"WARNING: isSymmetric() shoud be not called in program Green "
	       <<std::endl;
#endif
        Subscript i, j;

        for (i=0; i<this->m_; i++)
            for (j=i + 1; j<this->n_; j++)
                if(this->row_[i][j] != this->row_[j][i])
                    return false;
        return true;
    }
  
 
  }; // class SqMatrix



  /** Symmetric indefinite matrix in conventional (not packed)
      triangular upper storage: see NAG Fortran Library, Manual F07
      Sec. 3.3. Note that row-oriented storage is now indistinguishable from
      column-oriented one, allowing to use Lapack subroutines on a C-matrix.
  */
  template <class T> 
  class SyMatrix : public TNT::SqMatrix<T>
  {

  public:
    SyMatrix() : TNT::SqMatrix<T>() {}

    SyMatrix(const SyMatrix<T> &A) : SqMatrix<T>(A) {assert(TNT::SqMatrix<T>::isSymmetric());}
  
    SyMatrix(Subscript M, Subscript N, const T& value = T()) :
      SqMatrix<T>(M,N, value) {assert(this->isSymmetric());}
   
    SyMatrix(Subscript M, Subscript N, const T* v):  
      SqMatrix<T>(M,N, v)  {assert(this->isSymmetric());}

    SyMatrix(Subscript M, Subscript N, const char *s) : 
      SqMatrix<T>(M,N,s)  {assert(this->isSymmetric());}


    SyMatrix<T>& operator=(const T& scalar)
    { 
      this->set(scalar); 
      return *this;
    }
    // operator=: see operator= for class SqMatrix
    
    bool invert()
    { // matrix inversion
      // Double-precision complex 
      assert(this->m_ >=1 );

      Fortran_integer N = this->num_rows();
      TNT::Vector<Fortran_integer> index(N); // pivot
      Fortran_integer info = 0; // info on the success of operation
      Fortran_integer worksize = 5*N; //%% long enough ??
      char uplo = 'L';  
      // the argument UPLO set to L ("lower") corresponds to work on the
      // upper triangle of a symmetric row-oriented C matrix (while
      // would say to work on the lower triangle in the case a
      // column-oriented Fortran matrix)

      TNT::Vector<T> work(worksize, T());

      //std::cout<<"Typeid: "<<typeid(T()).name()<<endl;
      if(typeid(T()) == typeid(Fortran_double())) {//typeid: Sec. 15.4.5
        F77_DSYTRF(&uplo, &N, reinterpret_cast<Fortran_double*>(this->v_), &N,
               &index(1), reinterpret_cast<Fortran_double*>(&(work(1))),
               &worksize, &info); // LU factorization

        // reinterpret_cast necessary to skip static control of type
        // consistency while compiling, see Sec. 6.2.7; alternatively, set
        // to void* the type of the argument in the function declaration

        F77_DSYTRI(&uplo, &N, reinterpret_cast<Fortran_double*>(this->v_), &N,
               &index(1), reinterpret_cast<Fortran_double*>(&(work(1))),
               &info); // matrix inversion
      }
      else if(typeid(T() ) == typeid(Fortran_complex()) ) { 
        F77_ZSYTRF(&uplo, &N, reinterpret_cast<Fortran_complex*>(this->v_), &N,
            &index(1), reinterpret_cast<Fortran_complex*>(&(work(1))),
            &worksize, &info);  // LU factorization

        F77_ZSYTRI(&uplo, &N, reinterpret_cast<Fortran_complex*>(this->v_), &N,
            &index(1), reinterpret_cast<Fortran_complex*>(&(work(1))),
            &info); // inversion
      }
      else { // none of types implemented
        //cerr<<"ERROR in invert(): not implemented matrix inversion for type "
        //   << typeid(T()).name()<<endl; // name(): Sec. 15.4.4
          throw std::runtime_error("Not implemented matrix inversion for this type yet");
      }

      fill_lower_triangle();
      // after inversion, the lower triangle of the matrix must be filled
      // with values in the upper triangle
    
      return (info == 0 ? true: false);
    } // invert()

 
    void fill_lower_triangle()
    { // Fill the lower triangle of a symmetric matrix with values in
      // the upper triangle.
    
        for (Subscript i = 0; i < this->m_; i++) {
            for (Subscript j = 0; j < i; j++) {
                this->row_[i][j] = this->row_[j][i];
            }
        }
        return;
    } //  fill_lower_triangle()

 
  }; // class SyMatrix


} // namespace TNT 

