/*
This software was developed by Alessio Del Monte and Nicola Manini. It is not
subject to copyright protection and is in the public domain: permission is
granted to any individual or institution to use, copy, modify or redistribute
it. The authors make no guarantees about this software and assume no
responsibility for its use by other parties.

Whoever makes use of it is asked to cite "A. Del Monte, N. Manini, L.G.
Molinari, and G.P. Brivio, Mol. Phys. 103, 689 (2005)" and the URL
http://materia.fisica.unimi.it/manini/ivr.html.

This license statement should be provided in derived software.
*/

#include <iostream>
#include <cmath> // providing sqrt()
#include <cassert>
#include <string> // for State_base::State_base(char* str)
#include <sstream> // for State_base::State_base(char* str)
#include <algorithm> // for equal(), lexicographical_compare()
#include <valarray>

#include "global.hpp"


// Needed by class Tensor
#include <fstream> 
#include <sstream>
#include <map>

#include "lib_algorithms.hpp" // for factorial()
#include "tensor.hpp"

#include "state.hpp"

using namespace std;

namespace opt {
  extern bool zpe;
}

// static member variables
//
dimensionT State_base::dimension_ = 0; 
double* State_base::squaroot = NULL; 



State_base::State_base(char const* str){ 
  assert (dimension_ > 0);

  mode_ = new quantumT[dimension_];

  istringstream iss(str);
  long int vtmp=0;

  dimensionT i = 0; 
  while(i < dimension_ && iss >> vtmp) { 
    if(vtmp < 0) {
      cerr<<"ERROR in State_base(char*): negative number of phonons"<<endl; 
      exit(1);
    } 
    else if(vtmp> OVERT_MAX) {
      cerr<<"ERROR in State_base(char*): negative number of phonons"<<endl; 
      exit(1);
    }
    mode_[i]=vtmp;
    i++;
  }
  for( ; i< dimension_; i++) // remaining modes set to zero
    mode_[i] = quantumT();
} 

void State_base::set_squaroot()
{
  squaroot = new double[OVERT_MAX+1];

  for (quantumT i = 0; i <= OVERT_MAX; i++)
    squaroot[i] = double(sqrt(double(i))); 
} 


double State_base::perturbate(std::valarray<dimensionT> index, binaryT binary)
{ // perturbation elementary operator 
  double coeff = 1.0; //  coefficient

#ifdef GREEN_CHECK 
  binaryT maxBinary =  binaryT(std::pow(2.0, int(index.size() )) + 0.5 );
  assert(binary < maxBinary ); 
#endif
 
  std::size_t degree =  std::size_t(index.size());
  // size_t defined in <cstddef>, see Sec. 16.1.2; avoided reiterated
  // calls of size() and implicit cast in loop
    for (std::size_t k = 0; k < degree; k++) // 
    { 
#ifdef GREEN_CHECK
      assert(squaroot != NULL);
      assert(index[k] < dimension_);
      assert(index.size() < sizeof(binaryT) * 8 ); // max number of bit
#endif
      quantumT* p = &mode_[index[k]];
     
      if ((binary>>k) & 1) { 
#ifndef GREEN_AVOID_OVERT_MAX_CHECK 
	// check can be avoided if OVERT_MAX is high enough
	if (*p == OVERT_MAX) {
	  std::cerr<<"ERROR: exceeded the maximum quantum number "
		   <<OVERT_MAX<<endl;
	  exit(1);
	}
#endif 
	coeff *= squaroot[++(*p)]; 
      }
      else {
	if (*p == 0) return 0.0;
	coeff *= squaroot[(*p)--]; // NB: note post increment
      }
    } // loop on k 
  return coeff;
}

//*****  Friend functions 

ostream& operator<<(ostream& ostr, const State_base& st)
{ 
  ostr <<'|';
  for (dimensionT i = 0; i<State_base::dimension_-1; i++) 
    ostr<<int(st.mode_[i])<<' ';

  ostr<<int(st.mode_[State_base::dimension_-1])
      <<'>';
  return ostr;
}

NormalModeExpansion State::APESurface;


double State::energy() const 
{
#ifdef GREEN_CHECK
  assert(dimension_>0); assert(APESurface.count_modes()>0);
#endif
  double en = 0; 
  double zpe = (opt::zpe? 0.5 : 0.0);

  NormalModeExpansion::const_iterator pC;
  
  pC = APESurface.begin();
  for(dimensionT i = 0; i < dimension_; i++) { // non int: usare S
    en += (mode_[i] + zpe ) * pC->second ;
    pC++; 
  } 
  return en;
}


std::pair<NormalModeExpansion::const_iterator,
	  binaryT> SuccessorOperator_base::cursor; 
