/*
This software was developed by Alessio Del Monte and Nicola Manini. It is not
subject to copyright protection and is in the public domain: permission is
granted to any individual or institution to use, copy, modify or redistribute
it. The authors make no guarantees about this software and assume no
responsibility for its use by other parties.

Whoever makes use of it is asked to cite "A. Del Monte, N. Manini, L.G.
Molinari, and G.P. Brivio, Mol. Phys. 103, 689 (2005)" and the URL
http://materia.fisica.unimi.it/manini/ivr.html.

This license statement should be provided in derived software.
*/


/******************************************************************
                      QUANTUM STATE

1. Quantum state of a molecule as a dinamic array whose entries are
the occupation numbers of corresponding normal modes, while the number
of normal modes (i.e. the dimension of the quantum system) is saved in
a static variable.

2. Method for arbitrary-order elementary perturbation operator, see %%
   Sec. 2.2.

******************************************************************/

#ifndef GREEN_DIMENSION_T
std::cerr <<"WARNING[state.hpp]: GREEN_DIMENSION_T not defined: using default"
          <<endl;
#define GREEN_DIMENSION_T unsigned short int
#endif


typedef GREEN_DIMENSION_T dimensionT; 
// Dimension of the quantum system, i.e. the number of normal modes. 

typedef unsigned short int quantumT;
/* Type for the number of vibrational quanta (occupation number),
Sec. 2.2.3. If the memory and/or time requirements are critical, the
use of 'unsigned char' could be preferred, if the limit of 255 is
sufficient. */ 

typedef unsigned char binaryT; // elementary perturbation operator
			       // `shape': see perturbate() below



#ifndef OVERT_MAX
#define OVERT_MAX 65530
#endif

/* Maximum number of quanta of excitation in each oscillator. The
number must be less than the limit corresponding to the type chosen
for quantumT, see "limits.h". USHRT_MAX = 65535.  To change default
value, compile with -DOVERT_MAX=...  Pay attention to the vector of
squared integer up to OVERT_MAX in the class State_base.
*/

class State_base {

  friend bool operator==(const State_base& a, const State_base& b) {
    return std::equal(a.mode_, a.mode_ + State_base::dimension_, b.mode_);
  } // bool equal(In first, In last, In2first2), Sec. 18.5.4

  friend bool operator<(const State_base& a, const State_base& b) {
    return std::lexicographical_compare(a.mode_, a.mode_ + State_base::dimension_,
					b.mode_, b.mode_ + State_base::dimension_);
  } // lexicographical_compare(): see Sec. 18.9  
 
  friend std::ostream& operator<<(std::ostream&, const State_base&);
  // state printed with Ket notation

public:

  // Constructors
  //
  
  State_base(){ mode_ = new quantumT[dimension_]; }
    
  State_base(char const* c_string); //input from a string of the type 
                               // "2 1 0 0 1 "

  State_base(const State_base& st) { 
    mode_ = new quantumT[dimension_];
    std::copy(st.mode_, st.mode_ + dimension_, mode_); // copy: Sec. 18.6.1
  } 

  
  virtual ~State_base() { delete[] mode_; } // destructor
 
  State_base& operator= (const State_base& st) { 
    std::copy(st.mode_, st.mode_ + dimension_, mode_); // copy: Sec. 18.6.1
    return *this;
  } 

  // Access functions
  //
  quantumT& operator[] (dimensionT i) const {return mode_[i];} 
  quantumT mode(dimensionT i) const {return mode_[i];} // read only

  static dimensionT dimension() {return dimension_;}

  // Settings
  // 
  static void dimension(dimensionT d ) {dimension_ = d;}
  static void set_squaroot(); // see member squaroot below 

 
  // Perturbation elementary operator of arbitrary order
  //
  double perturbate(std::valarray<dimensionT> mIndex, // multi-index
			binaryT bin); // sequence of `a' and `a daggar'
  //

 /*
  Method emulating the change of quantum numbers of a state by the
  action of a product of destroy and create operators. The sequence of
  modes to be changed is given by the entries in the multi-index,
  while the sequence of `a' and `a daggar' corresponds to the
  positions of 0 and 1 respectively in the binary representation of
  parameter `bin'.

  Returns the perturbation quantum coefficient given by the product of
  square of quantum numbers, see Thesis Sec. 2.3.
*/

protected:

  quantumT* mode_; // occupation numbers

  static dimensionT dimension_; // number of normal modes, i.e. of
                                // vibrational degrees of freedom

  static double* squaroot;
  // square roots of the first OVERT_MAX integer numbers: for better
  // performance in perturbate() method while calculating the
  // coefficient to be returned
  
}; // class State_base



/*
 Given a certain parent state, SuccessorOperator_base gives all the
 possible states generated by means of anharmonic perturbation,
 according the parameters in the specified APES.
*/
class SuccessorOperator_base {
  
protected:

  const NormalModeExpansion& APESurface; // Input:
  const State_base parentS; 
  // if it was a reference, `sonS = parentS' would not copy the State_base

  static std::pair<NormalModeExpansion::const_iterator, binaryT> cursor;
  // cursor on all the possible combinations for elementary
  // perturbation operators

public:

  SuccessorOperator_base(const NormalModeExpansion& pAPESurface, 
			 const State_base& pParentS) : 
    APESurface (pAPESurface), parentS(pParentS) {
    
    // initial position of the cursor
    //
    cursor.first =  APESurface.begin_anh(); // first anharmonic coefficient
    cursor.second = binaryT(); 
  }

  bool thereIsNext() {
    return cursor.first != APESurface.end(); 
  } 
    
  std::pair<double, State_base> next_generated() {
    
    State_base sonS = parentS;
#ifdef GREEN_CHECK
    assert(thereIsNext()); 
#endif
    double perturbation_coeff = cursor.first->second;// anharmonic coefficient

    perturbation_coeff *= sonS.perturbate(cursor.first->first, cursor.second);
    

    // Move the cursor forward
    //
    if(cursor.second < binaryT(std::pow(2.0, 
					int(cursor.first->first.size()))) - 1) 
      cursor.second++;
    else { 
      cursor.first++;
      cursor.second = 0;
    }

    return std::make_pair(perturbation_coeff, sonS);
  } //
  
  
}; // class SuccessoOperator_base 



class State : public State_base 
{

public:

  // Constructors
  //
  State(): State_base() {}

  State(char const* c_string): State_base(c_string) {}

  State(const State_base& S_base): State_base(S_base)  { } // for cast
  //

  static void initialize(const char* coefficients, const char* frequencies)
  {
    APESurface.get(coefficients, frequencies); // reference setted
    State_base::dimension(APESurface.count_modes());
    assert(dimension_ > 0); //
    State_base::set_squaroot();
  }
  
  double energy() const; // quantum harmonic energy of the state

// if we are interested to performance, it would be better to define a
// member function energy() in class State_base, with frequencies as member
// data



  static NormalModeExpansion APESurface; // adiabatic potential energy
					 // surface


}; // class State 



class SuccessorOperator : public SuccessorOperator_base{

public:
  SuccessorOperator(const State& pParentS) : 
    SuccessorOperator_base(State::APESurface, pParentS) { 
    
  }

  std::pair<double, State> next_generated() 
    {
      std::pair<double, State_base> duo;
      duo = SuccessorOperator_base::next_generated();
      return std::make_pair(duo.first, State(duo.second)); // casting to State
    }
}; // class SuccessorOperator
