/*
This software was developed by Alessio Del Monte and Nicola Manini. It is not
subject to copyright protection and is in the public domain: permission is
granted to any individual or institution to use, copy, modify or redistribute
it. The authors make no guarantees about this software and assume no
responsibility for its use by other parties.

Whoever makes use of it is asked to cite "A. Del Monte, N. Manini, L.G.
Molinari, and G.P. Brivio, Mol. Phys. 103, 689 (2005)" and the URL
http://materia.fisica.unimi.it/manini/ivr.html.

This license statement should be provided in derived software.
*/

#include <iostream>
#include <fstream> 
#include <sstream>
#include <vector>
#include <valarray> // for the multi-index
#include <map>
#include <algorithm> // min_element(), max_element(), lexicographical_compare()
#include <functional> // for bind2dn()
#include <cassert>
#include <stdexcept>

#include "global.hpp"

#include "lib_algorithms.hpp" // for factorial()
#include "lib_vector.hpp" 

#include "tensor.hpp"

using namespace std;

bool isAnharmonic(const std::pair<const Multindex, double>& duo)
{
  return (duo.first.size() > 2);
} // 

bool isHarmonic(const std::pair<const Multindex, double>& duo)
{
  return (duo.first.size() == 2);
} // 



void PolynomialExpansion::get(string filenames, string frequencies)
{

  vector<string> stringVec;  
  
  istringstream iss(frequencies.c_str());
  string name;  // skipping white spaces
  if(iss>>name) { // name of file with frequencies, skipping white
		  // spaces
    get_from_file_freq(name.c_str()); //  normal modes
  }

  stringVec = split<string>(filenames.c_str());
  for(std::size_t i = 0; i < stringVec.size(); i++) {
   
    get_from_file(stringVec[i]);
  }
}


void PolynomialExpansion::get_from_file(string filename) 
{
  ifstream infile(filename.c_str());
  if(!infile.is_open()) {
	  throw std::runtime_error("Cannot open input file \"" + filename + "\"");
  }

  string line; 
  while(getline(infile, line) && line.length() > 0) {
    pair<Multindex, double> buffpair;
    
    try {
      get_pair(line, buffpair); // returns true if gotten a coefficient

      std::sort(&buffpair.first[0],
		&buffpair.first[buffpair.first.size()]); // i <= j <= k... 

      if(!insert(buffpair).second) {
	// insert() returns false if the multi-index is already present,
	// Sec. 17.4.1.7
	std::cerr <<"WARNING: multi-index "<<buffpair.first
		  <<" repeated"<<endl;
      }
   
    } catch(Cannot_get_pair) {
      std::cerr <<"WARNING: failed input from line \""<<line
		<<"\" in file "<<filename<<endl;
    }
  } // while getline 
} // get_from_file()



void PolynomialExpansion::get_from_file_freq(string filename) 
{
  ifstream infile(filename.c_str());
  if(!infile.is_open()) {
	  throw std::runtime_error("Cannot open input file \"" + filename + "\"");
  }

  int n = 1;
  double freq;
  string line; 
  while(getline(infile, line) && line.length() > 0) {
    istringstream  iss (line.c_str());

    if(iss >> freq ) { // got a coefficient

      Multindex index(n, 2); // valarray is (n, n)
      n++; // number for next normal mode
      bool successfull = insert(std::pair<Multindex, double>(index, freq)).second;
      if (!successfull) {
    	  throw std::runtime_error("Cannot insert frequency of polynomial expansion");
      }
    }
    else {
      std::cerr <<"WARNING: failed input from line \""<<line
		<<"\" in file "<<filename<<endl;
    }
  } 
} // get_from_file_old()


void get_pair(string line, pair<Multindex, double>& duo) 
{
  vector<double> buffvec = split<double>(line.c_str());
  
  if(buffvec.size() < 2) 
    throw Cannot_get_pair();

  duo.second = buffvec.back();  

  buffvec.pop_back(); // the last number was the coefficient

  duo.first.resize(buffvec.size()); // Sec. 22.4.3
  std::transform(buffvec.begin(), buffvec.end(), &duo.first[0],
		 cast<double, indexT>); // casts double to indexT
  // transform:  Sec. 18.6.2
}


void NormalModeExpansion::check_normal_modes() 
{  
  indexT num = 0; // number of normal modes
   
  NormalModeExpansion::const_iterator pPhi = begin();  
  // const_iterator to be preferred for read-only access to elements,
  // Sec. 16.3.1

  assert(begin() != end() );
  assert(size()>0);
  if(begin()->first.size()  <= 1) {
    cerr <<"ERROR: term of degree 1" <<pPhi->first<<endl; 
   
    exit(1);
  }

  else if(begin()->first.size() !=2 ) {   
    cerr <<"ERROR: none term of degree 2 found"<<endl;
    exit(1);
  }
  
  //*** Count tensors with rank 2 and check there aren not
  //mixed terms
  
  while(pPhi->first.size() == 2) {
    if(pPhi->first[0] != pPhi->first[1]) { // i != j
      cerr <<"ERROR: polynomial expansion not in normal modes: " 
	"non-diagonal term " <<pPhi->first<<endl;
      exit(1); 
    }
  
    if(pPhi->first[0] <= 0) { // check for 1-base indexing
      cerr <<"ERROR: 1-base indexing is required, contrary to "
	   <<pPhi->first<<endl; exit(1); 
    }
    num++;
    pPhi++;
  } // loop of coefficients of degree 2


  //*** Check range of indexes not to exceed the number of normal
  // modes

  for(pPhi = begin(); pPhi!= end(); pPhi++) {
    indexT min = *std::min_element(&pPhi->first[0], 
				   &pPhi->first[pPhi->first.size()]);
    // see algorithms Sec. 18.9
    indexT max = *std::max_element(&pPhi->first[0],
				   &pPhi->first[pPhi->first.size()]);
    if(min < 1 || max > num) {
      std::cerr <<"ERROR: some index exceeds the number of normal modes in "
		<<pPhi->first<<endl;
      exit(1);
    }
  }
  
  //*** Shift from 1-base to 0-base indexing
  //

  NormalModeExpansion::iterator pPhi2 = begin(); // not const_iterator
  while(pPhi2 != end()) {
    Multindex tempI = pPhi2->first;
    for(size_t i = 0; i < pPhi2->first.size(); ++i)
    {
    	assert(pPhi2->first[i] > 0);
    	tempI[i] = (pPhi2->first[i]) - 1;
    }
    pair<Multindex, double> tempPair(tempI, pPhi2->second);
    NormalModeExpansion::iterator nextIt = pPhi2; ++nextIt;
    // We have to get and save the reference to the next item, since the pPhi2 iterator
    // will become invalid after the following call to erase()
    erase(pPhi2); // Sec. 17.4.1.7
    insert(tempPair);
    pPhi2 = nextIt;
  } 

}
/* 
   Note that, because of the construction method employed for the
   map, the tensors with multi-index of 2 elements (the smallest it
   is possible) are at the beginning and without repetitions.
*/

