/*
This software was developed by Alessio Del Monte and Nicola Manini. It is not
subject to copyright protection and is in the public domain: permission is
granted to any individual or institution to use, copy, modify or redistribute
it. The authors make no guarantees about this software and assume no
responsibility for its use by other parties.

Whoever makes use of it is asked to cite "A. Del Monte, N. Manini, L.G.
Molinari, and G.P. Brivio, Mol. Phys. 103, 689 (2005)" and the URL
http://materia.fisica.unimi.it/manini/ivr.html.

This license statement should be provided in derived software.
*/

#ifndef GREEN_DIMENSION_T
std::cerr <<"WARNING[tensor.hpp]: GREEN_DIMENSION_T not defined: using default"
          <<endl;
#define GREEN_DIMENSION_T unsigned short int
#endif


typedef GREEN_DIMENSION_T indexT;

/* Multi-index of arbitrary order, with indexes sorted in increasing
   order and the peculiar comparison operator< defined below
 */


class Multindex : public std::valarray<indexT> { // valarray: Sec. 22.4

  friend bool operator<(const valarray<indexT>& v1, const valarray<indexT>& v2)
    {
      
      if(v1.size() != v2.size())
	return (v1.size() < v2.size() ? true : false);
      else  // if here, the two vectors have the same size
	return  std::lexicographical_compare(&v1[0], &v1[v1.size()], 
					     &v2[0], &v2[v2.size()]);
    } // operator<

 public:

  Multindex() : std::valarray<indexT>() {}
  Multindex(const indexT& val, size_t n) : std::valarray<indexT>(val, n) {}
  

  // private:

  void sort() { std::sort(&operator[](0), &operator[](size()) ); }
};



template<class T, class R> 
R cast(const T& x) { return R(x); }


class Factor {// abstract classes: Sec. 12.3
public:
  virtual double factor(const Multindex& index) = 0;
};

class TaylorFactor : public Factor { // 1 / !degree

public:
  double factor(const Multindex& index) {
    return 1.0 / algo::factorial(index.size());
  }
 
}; // class TaylorFactor 
// Object function, Sec. 18.4 

class SchwartzFactor : public Factor { 
// number of distinguishable permutations of the multindex
public:
  
  double factor(const Multindex& index) {
    
    std::size_t permutations = 
      algo::number_of_permutations<indexT>(&index[0],
					   &index[index.size()]);
    return static_cast<double>(permutations); // static cast 
  }
}; // class SchwartzFactor



class QuantumFactor : public Factor { // Object function, Sec. 18.4 
public: 
  double factor(const Multindex& index) {
    return 1.0/ std::sqrt(std::pow(2.0, static_cast<int>(index.size()) ));
  }
};


template <class FactorFunction> 
class Renormalize: public FactorFunction {  
 
   std:: map<Multindex, double>& expansion;

public:

  /*explicit*/ Renormalize(std:: map<Multindex, double> pe) : expansion(pe) {} 
  void operator()(std::pair<const Multindex, double>& duo)   {
    duo.second *= FactorFunction::factor(duo.first);
  }
};

bool isHarmonic(const std::pair<const Multindex, double>& duo);
bool isAnharmonic(const std::pair<const Multindex, double>& duo); 



// Coefficients (tensors) of a generic polynomial expansion
class PolynomialExpansion: public std::map<Multindex, double>{
 
public:

  PolynomialExpansion() : map<Multindex, double>() {}

  PolynomialExpansion::iterator begin_rank(unsigned d) 
  {
    PolynomialExpansion::iterator p = begin();

    while(p ->first.size() != d && p != end() )
      p++;
    return p;
  } 
 
  void get(std::string filenames, std::string frequencies = "");
  
  void Taylor_renormalization() { 
   std::for_each(begin_anh(), end(), Renormalize<TaylorFactor>(*this));
  }
  // all _anharmonic_ potential coefficients are divided by (degree!)

  void Schwartz_renormalization() { 
   std::for_each(begin_anh(), end(), Renormalize<SchwartzFactor>(*this));
  }  

//  Multiplies each potential coefficient for the number of
//  distinguishable permutations of its indexes, according to the
//  Schwarz law.

  
  PolynomialExpansion::const_iterator begin_anh() const {
    return std::find_if(begin(), end(), isAnharmonic); 
  }
  
  PolynomialExpansion::iterator begin_anh() { //%% necessario?
    return std::find_if(begin(), end(), isAnharmonic); 
  }
 
private:
  void get_from_file(std::string filename);
  // input from a file with coefficients in the form "1 1 2 1342.2"
  // for each line

  void get_from_file_freq(std::string frequencies);
 // input file for frequencies reading the value of frequency for each
 // normal mode as the first element of each line

}; // class PolynomialExpansion

 
void get_pair(std::string line, std::pair<Multindex, double>& duo);

class Cannot_get_pair { // exception 
public:
  Cannot_get_pair() {}
};
// throwing exception is more safely than returning a boolean, since force
// to treat the failure of input



class NormalModeExpansion: public  PolynomialExpansion {

public:

  NormalModeExpansion () : PolynomialExpansion() {}
 
  void get(std::string filenames, std::string frequencies = "") { //[[ ev. virtual
    PolynomialExpansion::get(filenames, frequencies);
    check_normal_modes(); 
  }

  std::size_t  count_modes() {
    return std::count_if(begin(), end(), isHarmonic); // Sec. 18.5.3
  } //%%  trying to use negator: Sec. 18.4.4.4

  void quantum_renormalization() {
   std::for_each(begin_anh(), end(), Renormalize
		 <QuantumFactor>(*this));
    }  
  // potential anharmonic coefficients are further divided by sqrt(2 ^
  // degree), according to the renormalization rule for quantum
  // operators (to do before A.I. search)
  
private:
  
  void check_normal_modes(); 
  // checks for normal modes, changes from 1-base to 0-base indexing 

};

template <class K, class T, class Cmp>
std::ostream& operator<<(std::ostream& ostr,  std::map<K, T, Cmp>& ma)
{ 
  typename std::map<K, T, Cmp>::const_iterator pM = ma.begin();
  // typename: see Sec. C.13.6
  // const_iterator: the key is a const type

  for( ; pM != ma.end(); pM++) {
    ostr <<'('<< pM->first<<' '<<pM->second<<')';
  }
  return ostr;
}

