/*
This software was developed by Alessio Del Monte and Nicola Manini. It is not
subject to copyright protection and is in the public domain: permission is
granted to any individual or institution to use, copy, modify or redistribute
it. The authors make no guarantees about this software and assume no
responsibility for its use by other parties.

Whoever makes use of it is asked to cite "A. Del Monte, N. Manini, L.G.
Molinari, and G.P. Brivio, Mol. Phys. 103, 689 (2005)" and the URL
http://materia.fisica.unimi.it/manini/ivr.html.

This license statement should be provided in derived software.
*/

#include <iostream>
#include <fstream> 
#include <string>
#include <sstream>

#include <vector>
#include <valarray>
#include <list> // Tiers
#include <map> // For tier construction

#include <algorithm> // sort e binary_search
#include <functional> // bind2nd(), plus(), mem_fun
#include <numeric> // for inner_product()
#include <complex>
#include <cassert>

#include "tnt_cmat_modified.h"

#include "global.hpp"

#include "lib_vector.hpp" 
#include "lib_algorithms.hpp" // factor 

#include "tensor.hpp"
#include "state.hpp"
#include "tier.hpp"

using namespace std;

namespace opt {
  extern bool zpe;
  extern bool square_ccs;
  extern bool sum_ccs;
  extern bool show_weights;
}

void Tier::print_verbose()
{
  for(size_t i = 0; i < size(); i++) 
    {
      ccout <<"- State n. "<<i + 1 <<" is "<<operator[](i)<<" with energy "
	    <<operator[](i).energy()<<"\n";
    }
}

TierBasis::TierBasis(const vector<string>& bright) : list<Tier>() 
  { 
    for(std::size_t i = 0; i < bright.size(); i++){ 
      double weight;
      istringstream  iss(bright[i].c_str());
      iss>> weight;
      
      string restofline; getline(iss, restofline);
      State tempS(restofline.c_str());
      WS.insert(make_pair(weight, tempS));
    }

    assert(!WS.empty());
   
    download();

    Tier::const_iterator pTier;
    pTier = std::unique(back().begin(), back().end()); // here front()==back()  
    // unique: Sec. 18.6.3   
    if(pTier != back().end())
      { cerr << "ERROR: repeated states for tier 0 from input"<<endl;
      exit(1); } 

  } // TierBasis::TierBasis()


bool TierBasis::next_tier(unsigned max_tier_size) 
  {
    WeightSortedMap::const_iterator pWS;
   
    for(pWS = WS.begin();  pWS != WS.end(); pWS++) {
     
      SuccessorOperator successorOp(pWS->second);
      
      while(successorOp.thereIsNext()) {

	pair<double, State> perturbation;
	perturbation = successorOp.next_generated();

	if(perturbation.first != 0.0 // parturbation succeeded
	   && !isThere(perturbation.second)) // basis orthogonality
	  { 

// cumulative (effective) coupling strength (c.c.s.), see Thesis,
// Sec. 3.3.3
	    double ccs; 

	    ccs = perturbation.first // 
	      / std::abs(perturbation.second.energy() - pWS->second.energy());

	    ccs = filter(opt::square_ccs ? ccs * ccs : ccs); // filtering 
	    ccs *= pWS->first; // multiplied for c.c.s. of the parent state

	    pair<StateSortedMap::iterator, bool> ins_info =
	      SW.insert(make_pair(perturbation.second, ccs)); 
	    //insert(): in such a manner we avoid searching the State
	    //for two times; see Sec. 17.4.1.7
	  
	    if(!ins_info.second) { // insert failed: the state was
				   // already collected in the map SW
	      if(opt::sum_ccs) { 
		ins_info.first->second +=  ccs; 
	      } 
	      else { // max of c.c.s. 
		ins_info.first->second = std::max(ins_info.first->second, 
						 ccs);
	      } // if insert in SW failed
	    
	    }

	} // if new state can be accepted
      } // loop on SuccessorOperator
    } // loop on working map WS

    WS.clear(); // 
    
    for(StateSortedMap::const_iterator pSW = SW.begin(); pSW != SW.end(); pSW++)
      // states are copied in the multi-map WS, automatically sorting
      // according their weight
      {
	WS.insert(make_pair(pSW->second, pSW->first));
      }

    download(max_tier_size);
    SW.clear();

    return true;
  } // TierBasis::next_tier()


void TierBasis::download(unsigned max_tier_size)
  {
    unsigned size = (max_tier_size != 0 ? //if zero no maximum fixed
		     std::min((size_t)max_tier_size, WS.size()) : WS.size());
        
    push_back(Tier(size)); 
    
    WS.resize(size); // necessary to resize WS if max_tier_size < size of WS
   
    if(opt::show_weights) {
      ccout <<"--- C.C.S. of states selected in the current tier:"<<newlc
	    <<WS<<newlc<<endl;
    }

    std::transform(WS.begin(), WS.end(), back().begin(), 
		   second_of_pair<double, State>); //  of a pair
     
    std::sort(back().begin(), back().end()); 
    // lexicographical ordering of states in the tier

  } // download()



ostream& operator<<(ostream& ostr, const TierBasis& basis) 
  // ev. generico template su liste
{
  for(TierBasis::const_iterator iter = basis.begin(); 
      iter != basis.end(); iter++)
    ostr << CMM << *iter << "\n";
  return ostr;
}


ostream& operator<<(ostream& ostr, const WeightSortedMap& multimap_)
  // ev. generico template
{
  PrintElement<std::pair<const double, State> > out;
  std::for_each(multimap_.begin(), multimap_.end(), out);
  
  return ostr;
} 

void addVij(TNT::Matrix<double>& M, const Tier& Tk, const Tier& Tk1) 
{ // <k | V | k> and <k + 1 | V | k+1>
 
  Tier::const_iterator pTk;
  for(pTk = Tk.begin();  pTk != Tk.end(); pTk++) {
   
    SuccessorOperator successorOp(*pTk);
	
    while(successorOp.thereIsNext()) {

      pair<double, State> perturbation;
      perturbation = successorOp.next_generated();

      if(perturbation.first != 0.0) // parturbation succeded
	{  
	  Tier::const_iterator pTk1;
	  
	  pTk1 = std::lower_bound(Tk1.begin(), Tk1.end(), perturbation.second); 
	  // binary search with lower_bound(): see Sec. 18.7.2
	      
	  if (pTk1 != Tk1.end() && *pTk1 == perturbation.second) 
	    { 
	      int indx_i = pTk - Tk.begin();
	      int indx_j = pTk1 - Tk1.begin();
		
	      M[indx_i][indx_j] += perturbation.first;
		
	    }
	} // 
    } //  while(successorOp.thereIsNext())  
  
  }// loop on first tier
  
} // addVij()


SBTMatrix::SBTMatrix(const TierBasis& basis)  : list<Block>() 
{
  TierBasis::const_iterator pBasis;
  
  for(pBasis = basis.begin(); pBasis != basis.end(); pBasis++)
    {
      
      TNT::Matrix<double> M_temp(pBasis->size(), pBasis->size());
      
      addVij(M_temp, *pBasis, *pBasis); // <k| V |k>
      
      for(int i = 0; i < M_temp.num_rows(); i++)
	M_temp[i][i] += (*pBasis)[i].energy();
      
      Block block; 
      
      push_back(block); 
      
      back().diag() = SyPotMatrix(M_temp);
      
      TierBasis::const_iterator pNext = pBasis; pNext++;
      
      if(pNext != basis.end()) {
	TNT::Matrix<double> M_temp2(pBasis->size(), pNext->size());
	
	addVij(M_temp2, *pBasis, *pNext);
	back().overdiag() = PotMatrix(M_temp2);
      }      
    }
} // SBTMatrix::SBTMatrix(const TierBasis& basis) 

