/*
This software was developed by Alessio Del Monte and Nicola Manini. It is not
subject to copyright protection and is in the public domain: permission is
granted to any individual or institution to use, copy, modify or redistribute
it. The authors make no guarantees about this software and assume no
responsibility for its use by other parties.

Whoever makes use of it is asked to cite "A. Del Monte, N. Manini, L.G.
Molinari, and G.P. Brivio, Mol. Phys. 103, 689 (2005)" and the URL
http://materia.fisica.unimi.it/manini/ivr.html.

This license statement should be provided in derived software.
*/

#include <list>
#include <vector>
#include <map>


#include "lib_sparsemat.hpp"

template<class T>
T filter(T x) {
  return 1 / sqrt(1 + 1 / (x*x));
}
// filter function for the weight of states (0 < weight <= 1),



/* 
   A tier, i.e. a collection of zero-th order states. These are
   ordered lexicographically to allow binary search.
 */
class Tier: public std::vector<State>
{
public:

  Tier(unsigned num) : std::vector<State>(num) {}
  bool isThere(const State& S) const {
	  return std::binary_search(begin(), end(), S);
  }
  
  void print_verbose();
}; // class Tier


typedef std::map<State, double> StateSortedMap;

class  WeightSortedMap : public std::multimap<double, State, std::greater<double> >
{//  working vector: states are sorted for weight (in decreasing order).
public:
  
  void resize(std::size_t newsize) 
  {
    std::multimap<double, State, std::greater<double> >::iterator pMap = begin();
    for(unsigned i = 0; i < newsize; i++) 
      pMap++; // do nothing
    erase(pMap, end());  
  }
};

/* 
   Tier basis of zero-order states.
*/
class TierBasis: public std::list<Tier>
{

public:
  
  TierBasis(const std::vector<std::string>& bright);

  ~TierBasis() {}
  // Destroyer defined just to avoid the following warning message when the options -Winline and -O3 are used:
  // tier.hpp:47:7: warning: inlining failed in call to 'TierBasis::~TierBasis()': call is unlikely and code size would grow [-Winline]

  bool isThere(const State& S)
  {
    
    for(TierBasis::const_iterator pTier = begin(); pTier != end(); pTier++) {
      if(pTier->isThere(S))
	return true; 
    }
    return false;
  }

  bool next_tier(unsigned max_tier_size = 0);
  /* If it is fixed a non-zero maximum size for the tier, states are
     selected according their effective cumulative coupling strength
     (c.c.s.), given by the formula:

     $w_b=\sum_{a} filter((\langle b|V|a\rangle / [E_a-E_b])^2) w_a$ 

     Setting the flag opt::sum_ccs to false, it is possible to change
     this algorithm, with \sum instead than \max; setting
     opt::square_ccs to false, the power is avoided.
  */
private: 

  WeightSortedMap WS; // working space
  StateSortedMap SW; //  

  void download(unsigned max_tier_size = 0); 

}; // class TierBasis


std::ostream& operator<<(std::ostream& ostr, const TierBasis& basis);


template<class Key, class Value> 
Value second_of_pair(const std::pair<Key, Value>& duo) { return duo.second;}
// Sec. 18.6.2,


template<class Key, class Value>
std::ostream& operator<<(std::ostream& ostr, const std::pair<const Key, Value> duo)
{
  return  ostr << "("<<duo.first<<" "<<duo.second<<")";
} //

template<class T>//, class Out) // move from here
class PrintElement // 18.4 and 18.5.1
{
public:
  void operator()(T x){ std::cout << x; }// Out << x;}
};


std::ostream& operator<<(std::ostream& ostr, const WeightSortedMap& multimap_);

// add the matrix <T_i|V|T_j> to the given matrix
void addVij(TNT::Matrix<double>& M, const Tier& Tk, const Tier& Tk1);
/* 
NOTE : The simmetry of <k|V|k> is guaranteed by the fact that if state
|i> is generated from a state |j> by the elementary perturbation
operator Op, then the state |j> is in turn generated from |i> by
complementary operator !Op (corrisponding to the bit-to-bit negated
binary number this implementation)
*/


typedef sparse::Matrix<double> PotMatrix;  //[[rename
typedef sparse::SyMatrix<double> SyPotMatrix; 

class Block { 
public:
  Block() {}
  ~Block() {} // See notes above for defining ~TierBasis()

  Block& operator=(const Block& B) 
  { 
    diagonal = B.diagonal; overdiagonal = B.overdiagonal;
    return *this;
   }

  SyPotMatrix& diag()  {return diagonal;}
  const SyPotMatrix& diag() const  {return diagonal;} 
  PotMatrix& overdiag()   {return overdiagonal;}
   const PotMatrix& overdiag()  const {return overdiagonal;}

private:
  SyPotMatrix diagonal; // <k | V | k>
  PotMatrix overdiagonal; // <k | V | k + 1>

};


class CmpDiag { // Predicate: Sec. 18.4.2
public: 
  bool operator()(const Block& x,const  Block& y) const  {
    return x.diag().num_rows() <  y.diag().num_rows();
  }
};

class CmpOverdiag { // Predicate: Sec. 18.4.2

public: 
  bool operator()(const Block& x,const  Block& y) const  {
    return x.overdiag().num_rows() * x.overdiag().num_cols() 
      <  y.overdiag().num_rows() * y.overdiag().num_cols();
  }
};

/****

   Symmetric block-tridiagonal matrix, as it is the case of the matrix
   representation of the Hamiltonian operator H in the tier basis, by
   construction of tiers.

   This is really true only if we do not alter the search technique of
   the tier basis by imposing a maximum size to each tier. If it is the
   case, we must neglect couplings between states belonging to non-adjacent
   tiers.
*/

class SBTMatrix : public std::list<Block> {

public:
  
  //SBTMatrix() : list<Block>() {}
  SBTMatrix(const TierBasis& basis);


  unsigned max_diagonal_capacity()  { 
    SBTMatrix::iterator pBlock;
    pBlock = std::max_element(begin(), end(), CmpDiag() );
    unsigned rows  = pBlock->diag().num_rows();
    return rows * rows;
  }

 unsigned max_overdiagonal_capacity()  { 
    SBTMatrix::iterator pBlock;
    pBlock = std::max_element(begin(), end(), CmpOverdiag() );
    return pBlock->overdiag().num_rows() *  pBlock->overdiag().num_cols();
  }


}; // class  SBTMatrix

				  
